/**
 * Tax calculation utilities
 * Saudi VAT rate is 15%
 */

const VAT_RATE = 0.15;

/**
 * Calculate VAT (15%) for a given amount
 * @param {number} amount - The base amount to calculate VAT for
 * @returns {number} - The calculated VAT amount
 */
export const calculateVAT = (amount) => {
  if (!amount || amount <= 0) return 0;
  return amount * VAT_RATE;
};

/**
 * Calculate total amount including VAT
 * @param {number} baseAmount - The base amount before VAT
 * @returns {number} - The total amount including VAT
 */
export const calculateTotalWithVAT = (baseAmount) => {
  if (!baseAmount || baseAmount <= 0) return 0;
  return baseAmount + calculateVAT(baseAmount);
};

/**
 * Get correct tax amount from price details
 * Validates if the provided tax is reasonable (at least 10% of expected VAT)
 * If not, calculates the correct VAT
 * 
 * @param {Object} priceDetails - Price details object
 * @param {number} basePrice - Base price amount
 * @param {string} type - Type of booking ('rental' or 'limousine')
 * @returns {number} - The correct tax amount
 */
export const getCorrectTaxAmount = (priceDetails, basePrice, type = 'rental') => {
  if (!basePrice || basePrice <= 0) return 0;
  
  const expectedVAT = calculateVAT(basePrice);
  const taxKey = type === 'limousine' ? 'vat' : 'tax';
  const providedTax = priceDetails?.[taxKey] || 0;
  
  // If provided tax is less than 10% of expected VAT, use calculated VAT
  if (providedTax < (expectedVAT * 0.1)) {
    return expectedVAT;
  }
  
  return providedTax;
};

/**
 * Format tax percentage for display
 * @returns {string} - Formatted tax percentage
 */
export const getTaxPercentageText = () => {
  return `${VAT_RATE * 100}%`;
};

/**
 * Check if tax amount seems correct (within reasonable range)
 * @param {number} taxAmount - The tax amount to validate
 * @param {number} baseAmount - The base amount
 * @returns {boolean} - True if tax seems reasonable
 */
export const isTaxAmountReasonable = (taxAmount, baseAmount) => {
  if (!baseAmount || baseAmount <= 0) return false;
  if (!taxAmount || taxAmount <= 0) return false;
  
  const expectedVAT = calculateVAT(baseAmount);
  const tolerance = 0.05; // 5% tolerance
  
  return Math.abs(taxAmount - expectedVAT) <= (expectedVAT * tolerance);
};
