/**
 * Utility functions for handling membership data from API
 */

/**
 * Sort memberships by startPoints in ascending order
 * @param {Array} memberships - Array of membership objects from API
 * @returns {Array} Sorted memberships array
 */
export const sortMembershipsByPoints = (memberships) => {
  if (!memberships || !Array.isArray(memberships)) return [];
  
  return [...memberships].sort((a, b) => {
    const pointsA = parseInt(a.startPoints) || 0;
    const pointsB = parseInt(b.startPoints) || 0;
    return pointsA - pointsB;
  });
};

/**
 * Get membership tier information based on points
 * @param {number} userPoints - User's current points
 * @param {Array} memberships - Sorted memberships array
 * @returns {Object} Current tier information
 */
export const getCurrentTier = (userPoints, memberships) => {
  if (!memberships || memberships.length === 0) {
    return {
      currentTier: null,
      nextTier: null,
      progressPercentage: 0,
      tierIndex: -1
    };
  }

  const sortedMemberships = sortMembershipsByPoints(memberships);
  
  // Find current tier
  let currentTier = null;
  let nextTier = null;
  let tierIndex = -1;
  let progressPercentage = 0;

  for (let i = sortedMemberships.length - 1; i >= 0; i--) {
    const tier = sortedMemberships[i];
    const startPoints = parseInt(tier.startPoints) || 0;
    
    if (userPoints >= startPoints) {
      currentTier = tier;
      tierIndex = i;
      
      // Calculate progress to next tier
      if (i < sortedMemberships.length - 1) {
        nextTier = sortedMemberships[i + 1];
        const currentTierEndPoints = parseInt(tier.endPoints) || startPoints;
        const nextTierStartPoints = parseInt(nextTier.startPoints) || currentTierEndPoints;
        
        const tierRange = nextTierStartPoints - startPoints;
        const pointsInTier = userPoints - startPoints;
        
        progressPercentage = tierRange > 0 
          ? Math.min((pointsInTier / tierRange) * 100, 100)
          : 100;
      } else {
        // User is at the highest tier
        progressPercentage = 100;
      }
      break;
    }
  }

  return {
    currentTier,
    nextTier,
    progressPercentage,
    tierIndex
  };
};

/**
 * Get tier display data for UI
 * @param {Array} memberships - Sorted memberships array
 * @param {Function} t - Translation function
 * @returns {Array} Formatted tier data for display
 */
export const getTierDisplayData = (memberships, t) => {
  if (!memberships || memberships.length === 0) return [];
  
  const sortedMemberships = sortMembershipsByPoints(memberships);
  return sortedMemberships.map((membership, index) => ({
    id: membership.memberId,
    name: membership.memberName,
    arabicName: membership.arabicName,
    startPoints: parseInt(membership.startPoints) || 0,
    endPoints: parseInt(membership.endPoints) || 0,
    discountPercentage: parseFloat(membership.discountPercentage) || 0,
    maxDiscount: parseFloat(membership.maxDiscount) || 0,
    icon: membership.icon,
    details: membership.details,
    arabicDetails: membership.arabicDetails,
    // Additional properties for UI
    threshold: parseInt(membership.startPoints) || 0,
    value: membership.startPoints ? parseInt(membership.startPoints).toLocaleString() : "0",
    color: getTierColor(index, sortedMemberships.length)
  }));
};

/**
 * Get color for tier based on index
 * @param {number} index - Tier index
 * @param {number} totalTiers - Total number of tiers
 * @returns {string} Color hex code
 */
const getTierColor = (index, totalTiers) => {
  const colors = [
    "#CD7F32", // Bronze
    "#C0C0C0", // Silver  
    "#FFD700", // Gold
    "#E5E4E2"  // Platinum
  ];
  
  return colors[index] || "#fbbf24";
};

/**
 * Get handle state for tier slider
 * @param {number} handleIndex - Handle index (0, 1, 2)
 * @param {number} userPoints - User's current points
 * @param {Array} tierDisplayData - Formatted tier data
 * @param {Object} animations - Animation objects
 * @returns {Object} Handle state object
 */
export const getHandleState = (handleIndex, userPoints, tierDisplayData, animations) => {
  if (!tierDisplayData || tierDisplayData.length === 0) {
    return { icon: null, animation: null, color: "#BFBFBF" };
  }

  // Empty state
  if (userPoints === 0) {
    return { icon: null, animation: null, color: "#BFBFBF" };
  }

  // Get the tier for this handle
  const tier = tierDisplayData[handleIndex];
  if (!tier) {
    return { icon: null, animation: null, color: "#BFBFBF" };
  }

  const tierThreshold = tier.threshold;
  const nextTier = tierDisplayData[handleIndex + 1];
  const nextTierThreshold = nextTier ? nextTier.threshold : Infinity;

  // User has reached this tier
  if (userPoints >= tierThreshold) {
    // User is currently in this tier (between this tier and next)
    if (userPoints < nextTierThreshold) {
      // Show animation for current tier
      const animationKey = getAnimationKey(handleIndex, tierDisplayData.length);
      return {
        icon: null,
        animation: animations[animationKey] || null,
        color: "#fbbf24"
      };
    } else {
      // User has passed this tier, show checkmark
      return {
        icon: "check",
        animation: null,
        color: "#515151" // Green for completed tiers
      };
    }
  }

  // User hasn't reached this tier yet
  return { icon: null, animation: null, color: "#BFBFBF" };
};

/**
 * Get animation key for tier
 * @param {number} handleIndex - Handle index
 * @param {number} totalTiers - Total number of tiers
 * @returns {string} Animation key
 */
const getAnimationKey = (handleIndex, totalTiers) => {
  if (totalTiers <= 1) return null;
  
  const animationKeys = ['silver', 'gold', 'platinum'];
  return animationKeys[handleIndex] || null;
};

/**
 * Calculate progress width for slider
 * @param {number} userPoints - User's current points
 * @param {Array} tierDisplayData - Formatted tier data
 * @returns {number} Progress width percentage
 */
export const calculateProgressWidth = (userPoints, tierDisplayData) => {
  if (!tierDisplayData || tierDisplayData.length === 0) return 0;
  if (userPoints === 0) return 0;

  // المواضع الجديدة من اليمين (progress fills from right to left)
  // Position 73% from left = 27% from right (Bronze)
  // Position 48% from left = 52% from right (Silver)
  // Position 22% from left = 78% from right (Gold)
  // Position 0% from left = 100% from right (Platinum)
  
  const tierPositions = [
    { position: 27, start: tierDisplayData[0]?.startPoints || 0, end: tierDisplayData[0]?.endPoints || 0 },   // Bronze: 0 - 10,000
    { position: 52, start: tierDisplayData[1]?.startPoints || 0, end: tierDisplayData[1]?.endPoints || 0 },   // Silver: 10,001 - 25,000
    { position: 78, start: tierDisplayData[2]?.startPoints || 0, end: tierDisplayData[2]?.endPoints || 0 },   // Gold: 25,001 - 50,000
    { position: 100, start: tierDisplayData[3]?.startPoints || 0, end: tierDisplayData[3]?.endPoints || 0 }   // Platinum: 50,001+
  ];

  // حساب دقيق: نحدد في أي tier المستخدم موجود
  for (let i = 0; i < tierPositions.length; i++) {
    const tier = tierPositions[i];
    const prevTier = i > 0 ? tierPositions[i - 1] : { position: 0, start: 0, end: 0 };
    
    // التحقق: هل المستخدم في هذا الـ tier؟
    // المستخدم في الـ tier إذا كانت نقاطه >= start و <= end
    if (userPoints >= tier.start && userPoints <= tier.end) {
      // حساب النسبة المئوية داخل هذا الـ tier
      const tierRange = tier.end - tier.start;
      const userProgressInTier = userPoints - tier.start;
      const progressPercentage = tierRange > 0 ? (userProgressInTier / tierRange) : 1;
      
      // حساب الموضع البصري
      const prevPosition = i > 0 ? tierPositions[i - 1].position : 0;
      const positionRange = tier.position - prevPosition;
      const visualProgress = prevPosition + (progressPercentage * positionRange);
      
      console.log(`User Points: ${userPoints}, Tier: ${i}, Range: ${tier.start}-${tier.end}, Progress: ${progressPercentage.toFixed(2)}, Visual: ${visualProgress.toFixed(2)}%`);
      
      return Math.min(visualProgress, 100);
    }
    
    // إذا تجاوز المستخدم هذا الـ tier تماماً
    if (userPoints > tier.end && i < tierPositions.length - 1) {
      continue; // انتقل للـ tier التالي
    }
  }

  // إذا كانت النقاط أقل من أول tier
  if (userPoints < tierPositions[0].start) {
    return 0;
  }

  // إذا كانت النقاط أكبر من آخر tier
  if (userPoints > tierPositions[tierPositions.length - 1].end) {
    return 100;
  }

  return 0;
};
