/**
 * ثوابت و helpers خاصة بحجز الليموزين
 */

import { URL } from '../constants/api';
import cookieStorage from './cookieStorage';

// خريطة تحويل المواقع إلى أرقام
export const LOCATION_MAP = {
  JEDDAH_AIRPORT: 1,
  JEDDAH: 2,
  MAKKAH: 3,
};

// أنواع الحجز
export const RESERVATION_TYPES = {
  AIRPORT: 'AIRPORT',
  OTHER_PLACES: 'OTHER_PLACES',
};

// API Endpoints
export const API_ENDPOINTS = {
  AIRPORT: '/limousine-reservations/airport',
  OTHER_PLACES: '/limousine-reservations/other-places',
};

// القيم الافتراضية لـ "أماكن أخرى" - استخدام البيانات الفعلية من الصورة
export const OTHER_PLACES_DEFAULTS = {
  latitude: 21.484881536099145,
  longitude: 39.19270599312848,
  toLatitude: 21.484402335241622,
  toLongitude: 39.19262016244,
  promoCode: "",
  pointsToUse: 1073741824,
  servicesAddons: [],
};

/**
 * تحديد API endpoint حسب نوع الحجز
 */
export const getApiEndpoint = (reservationType) => {
  const isOtherPlaces = reservationType === RESERVATION_TYPES.OTHER_PLACES;
  
  const endpoint = isOtherPlaces 
    ? URL(API_ENDPOINTS.OTHER_PLACES)
    : URL(API_ENDPOINTS.AIRPORT);
    
  return endpoint;
};

/**
 * بناء البيانات حسب نوع الحجز
 */
export const buildBookingPayload = (bookingData) => {
  const isOtherPlaces = bookingData.reservationType === RESERVATION_TYPES.OTHER_PLACES;
  const isHourly = bookingData.reservationType === "HOURLY";
  
  if (isHourly) {
    return {
      carId: bookingData.carId,
      cityId: bookingData.cityId || bookingData.from,
      latitude: bookingData.latitude || -90,
      longitude: bookingData.longitude || -90,
      address: bookingData.address || "1skdjflsdkjfsd",
      hours: bookingData.hours || 80,
      promoCode: bookingData.promoCode || "",
      pointsToUse: bookingData.pointsToUse || 1073741824,
      notes: bookingData.notes || "string",
      reservationDate: bookingData.reservationDate,
      servicesAddons: bookingData.servicesAddons || []
    };
  } else if (isOtherPlaces) {
    return {
      carId: bookingData.carId,
      from: LOCATION_MAP[bookingData.from] || bookingData.from,
      to: LOCATION_MAP[bookingData.to] || bookingData.to,
      latitude: bookingData.latitude,
      longitude: bookingData.longitude,
      toLatitude: bookingData.toLatitude,
      toLongitude: bookingData.toLongitude,
      promoCode: bookingData.promoCode || "",
      pointsToUse: bookingData.pointsToUse || 1073741824,
      reservationDate: bookingData.reservationDate,
      servicesAddons: bookingData.servicesAddons || []
    };
  } else {
    return {
      carId: bookingData.carId,
      reservationDate: bookingData.reservationDate,
      flightNumber: bookingData.flightNumber || null,
      from: LOCATION_MAP[bookingData.from] || bookingData.from,
      address: bookingData.address || null,
      latitude: bookingData.latitude || null,
      longitude: bookingData.longitude || null,
      to: LOCATION_MAP[bookingData.to] || bookingData.to,
      promoCode: bookingData.promoCode || null,
      pointsToUse: bookingData.pointsToUse || null,
      servicesAddons: bookingData.servicesAddons || []
    };
  }
};

/**
 * التحقق من صحة بيانات الحجز
 */
export const validateBookingData = (bookingData) => {
  if (!bookingData) {
    return { isValid: false, error: 'بيانات الحجز مفقودة' };
  }

  if (!bookingData.carId) {
    return { isValid: false, error: 'معرف السيارة مفقود' };
  }

  if (!bookingData.reservationDate) {
    return { isValid: false, error: 'تاريخ الحجز مفقود' };
  }

  // للـ HOURLY نتحقق من cityId بدلاً من from/to
  if (bookingData.reservationType === "HOURLY") {
    if (!bookingData.cityId) {
      return { isValid: false, error: 'معرف المدينة مفقود' };
    }
  } else {
    // للأنواع الأخرى نتحقق من from/to
    if (!bookingData.from || !bookingData.to) {
      return { isValid: false, error: 'بيانات الموقع مفقودة' };
    }
  }

  return { isValid: true };
};

/**
 * تنظيف البيانات المحفوظة بعد الحجز
 */
export const cleanupBookingData = () => {
  if (typeof window !== 'undefined') {
    cookieStorage.removeItem('bookingLim');
    cookieStorage.removeItem('reservationPriceLimResult');
    cookieStorage.removeItem('carLimData');
  }
};

/**
 * بناء query parameters لصفحة الطلب
 */
export const buildOrderQueryParams = (result) => {
  return new URLSearchParams({
    reservationId: result.reservationId?.toString() || '',
    total: result.finalAmount?.toString() || '',
    paidAmount: result.paymentAmount?.toString() || '',
    totalDiscount: result.totalDiscount?.toString() || '',
    reservationDate: result.reservationDate || '',
    carName: result.carName || '',
    firstName: result.firstName || '',
    lastName: result.lastName || ''
  });
}; 
