'use client';

import { useEffect, useRef, useCallback, useState } from 'react';
import { 
  debounce, 
  throttle, 
  createIntersectionObserver,
  getConnectionSpeed,
  shouldLoadHighQuality 
} from '@/utils/performanceUtils';

/**
 * Hook للتحقق من ظهور العنصر في viewport
 * @param {object} options - خيارات Intersection Observer
 * @returns {[ref, isVisible]} - مرجع العنصر وحالة الظهور
 */
export function useIntersectionObserver(options = {}) {
  const [isVisible, setIsVisible] = useState(false);
  const elementRef = useRef(null);
  const observerRef = useRef(null);

  useEffect(() => {
    const element = elementRef.current;
    if (!element) return;

    const callback = (entries) => {
      entries.forEach((entry) => {
        if (entry.isIntersecting) {
          setIsVisible(true);
          if (options.once !== false) {
            observerRef.current?.disconnect();
          }
        } else if (!options.once) {
          setIsVisible(false);
        }
      });
    };

    observerRef.current = createIntersectionObserver(callback, {
      threshold: options.threshold || 0.1,
      rootMargin: options.rootMargin || '50px',
      root: options.root || null,
    });

    if (observerRef.current) {
      observerRef.current.observe(element);
    }

    return () => {
      observerRef.current?.disconnect();
    };
  }, [options.threshold, options.rootMargin, options.once, options.root]);

  return [elementRef, isVisible];
}

/**
 * Hook لاستخدام debounce مع React
 * @param {Function} callback - الدالة المراد تأخيرها
 * @param {number} delay - وقت التأخير بالميلي ثانية
 */
export function useDebounce(callback, delay = 300) {
  const callbackRef = useRef(callback);

  useEffect(() => {
    callbackRef.current = callback;
  }, [callback]);

  return useCallback(
    debounce((...args) => callbackRef.current(...args), delay),
    [delay]
  );
}

/**
 * Hook لاستخدام throttle مع React
 * @param {Function} callback - الدالة المراد تقييدها
 * @param {number} limit - الحد الزمني بالميلي ثانية
 */
export function useThrottle(callback, limit = 300) {
  const callbackRef = useRef(callback);

  useEffect(() => {
    callbackRef.current = callback;
  }, [callback]);

  return useCallback(
    throttle((...args) => callbackRef.current(...args), limit),
    [limit]
  );
}

/**
 * Hook للحصول على معلومات الاتصال
 */
export function useConnectionInfo() {
  const [connectionInfo, setConnectionInfo] = useState({
    speed: 'unknown',
    shouldLoadHQ: true,
  });

  useEffect(() => {
    setConnectionInfo({
      speed: getConnectionSpeed(),
      shouldLoadHQ: shouldLoadHighQuality(),
    });

    // الاستماع لتغيرات الاتصال
    if (typeof navigator !== 'undefined' && navigator.connection) {
      const updateConnection = () => {
        setConnectionInfo({
          speed: getConnectionSpeed(),
          shouldLoadHQ: shouldLoadHighQuality(),
        });
      };

      navigator.connection.addEventListener('change', updateConnection);
      return () => {
        navigator.connection.removeEventListener('change', updateConnection);
      };
    }
  }, []);

  return connectionInfo;
}

/**
 * Hook لتحميل البيانات بشكل كسول
 * @param {Function} fetchFn - دالة جلب البيانات
 * @param {object} options - خيارات التحميل
 */
export function useLazyLoad(fetchFn, options = {}) {
  const [data, setData] = useState(null);
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState(null);
  const [elementRef, isVisible] = useIntersectionObserver(options);
  const hasFetched = useRef(false);

  useEffect(() => {
    if (isVisible && !hasFetched.current) {
      hasFetched.current = true;
      setLoading(true);
      
      fetchFn()
        .then((result) => {
          setData(result);
          setError(null);
        })
        .catch((err) => {
          setError(err);
        })
        .finally(() => {
          setLoading(false);
        });
    }
  }, [isVisible, fetchFn]);

  return { data, loading, error, ref: elementRef };
}

/**
 * Hook لتتبع التمرير بشكل محسّن
 */
export function useOptimizedScroll() {
  const [scrollY, setScrollY] = useState(0);
  const [scrollDirection, setScrollDirection] = useState('down');
  const lastScrollY = useRef(0);

  useEffect(() => {
    let ticking = false;

    const updateScroll = () => {
      const currentScrollY = window.scrollY;
      
      setScrollY(currentScrollY);
      setScrollDirection(currentScrollY > lastScrollY.current ? 'down' : 'up');
      
      lastScrollY.current = currentScrollY;
      ticking = false;
    };

    const handleScroll = () => {
      if (!ticking) {
        window.requestAnimationFrame(updateScroll);
        ticking = true;
      }
    };

    window.addEventListener('scroll', handleScroll, { passive: true });
    
    return () => {
      window.removeEventListener('scroll', handleScroll);
    };
  }, []);

  return { scrollY, scrollDirection };
}

/**
 * Hook لقياس أداء المكون
 * @param {string} componentName - اسم المكون
 */
export function usePerformanceMonitor(componentName) {
  const renderCount = useRef(0);
  const mountTime = useRef(null);

  useEffect(() => {
    renderCount.current += 1;
    
    if (!mountTime.current) {
      mountTime.current = performance.now();
      console.log(`[Performance] ${componentName} mounted at ${mountTime.current}ms`);
    }

    return () => {
      if (mountTime.current) {
        const lifetime = performance.now() - mountTime.current;
        console.log(`[Performance] ${componentName} unmounted after ${lifetime.toFixed(2)}ms`);
        console.log(`[Performance] ${componentName} rendered ${renderCount.current} times`);
      }
    };
  });

  return {
    renderCount: renderCount.current,
    logRender: () => {
      console.log(`[Performance] ${componentName} render #${renderCount.current}`);
    },
  };
}

/**
 * Hook للتحميل المتأخر للصور
 * @param {string} src - مصدر الصورة
 * @param {object} options - خيارات التحميل
 */
export function useLazyImage(src, options = {}) {
  const [imageSrc, setImageSrc] = useState(options.placeholder || null);
  const [isLoaded, setIsLoaded] = useState(false);
  const [error, setError] = useState(null);
  const [elementRef, isVisible] = useIntersectionObserver({
    threshold: 0.01,
    rootMargin: '100px',
    ...options,
  });

  useEffect(() => {
    if (isVisible && !isLoaded && src) {
      const img = new Image();
      
      img.onload = () => {
        setImageSrc(src);
        setIsLoaded(true);
        setError(null);
      };
      
      img.onerror = () => {
        setError(new Error('Failed to load image'));
        setIsLoaded(false);
      };
      
      img.src = src;
    }
  }, [isVisible, src, isLoaded]);

  return { 
    ref: elementRef, 
    imageSrc, 
    isLoaded, 
    error 
  };
}

/**
 * Hook لتحسين البحث
 * @param {Function} searchFn - دالة البحث
 * @param {number} delay - وقت التأخير
 */
export function useOptimizedSearch(searchFn, delay = 500) {
  const [query, setQuery] = useState('');
  const [results, setResults] = useState([]);
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState(null);

  const debouncedSearch = useDebounce(async (searchQuery) => {
    if (!searchQuery.trim()) {
      setResults([]);
      return;
    }

    setLoading(true);
    setError(null);

    try {
      const data = await searchFn(searchQuery);
      setResults(data);
    } catch (err) {
      setError(err);
      setResults([]);
    } finally {
      setLoading(false);
    }
  }, delay);

  const handleSearch = (value) => {
    setQuery(value);
    debouncedSearch(value);
  };

  return {
    query,
    results,
    loading,
    error,
    handleSearch,
  };
}

/**
 * Hook لحفظ حالة في localStorage مع performance optimization
 * @param {string} key - مفتاح localStorage
 * @param {any} initialValue - القيمة الابتدائية
 */
export function useOptimizedLocalStorage(key, initialValue) {
  const [storedValue, setStoredValue] = useState(() => {
    if (typeof window === 'undefined') return initialValue;
    
    try {
      const item = window.localStorage.getItem(key);
      return item ? JSON.parse(item) : initialValue;
    } catch (error) {
      console.warn(`Error loading ${key} from localStorage:`, error);
      return initialValue;
    }
  });

  const setValue = useCallback(
    (value) => {
      try {
        const valueToStore = value instanceof Function ? value(storedValue) : value;
        setStoredValue(valueToStore);
        
        if (typeof window !== 'undefined') {
          window.localStorage.setItem(key, JSON.stringify(valueToStore));
        }
      } catch (error) {
        console.warn(`Error saving ${key} to localStorage:`, error);
      }
    },
    [key, storedValue]
  );

  return [storedValue, setValue];
}

/**
 * Hook للكشف عن حجم الشاشة
 */
export function useMediaQuery(query) {
  const [matches, setMatches] = useState(false);

  useEffect(() => {
    if (typeof window === 'undefined') return;

    const media = window.matchMedia(query);
    
    if (media.matches !== matches) {
      setMatches(media.matches);
    }

    const listener = (e) => setMatches(e.matches);
    media.addEventListener('change', listener);
    
    return () => media.removeEventListener('change', listener);
  }, [query, matches]);

  return matches;
}

/**
 * Hook للكشف عن الأجهزة المحمولة
 */
export function useIsMobileDevice() {
  return useMediaQuery('(max-width: 768px)');
}

export default {
  useIntersectionObserver,
  useDebounce,
  useThrottle,
  useConnectionInfo,
  useLazyLoad,
  useOptimizedScroll,
  usePerformanceMonitor,
  useLazyImage,
  useOptimizedSearch,
  useOptimizedLocalStorage,
  useMediaQuery,
  useIsMobileDevice,
};

