// hook/useLocationSelection.js
'use client';

import { useRouter } from '@/navigation';
import { setPickupLocation, setSavedEndDate, setSavedStartDate, setSavedStartTime, setSelectedBranchId, setSelectedCityId } from '@/store/features/rental/rentalSlice';
import cookieStorage from '@/utils/cookieStorage';
import { useSearchParams } from 'next/navigation';
import { useCallback } from 'react';
import { useDispatch, useSelector } from 'react-redux';

const useLocationSelection = (onLocationChange) => {
    const { startTime, selectedRange, duration, selectedCarTypes, deliveryType, selectedCityId, selectedBranchId } = useSelector((state) => state.rental);
    const searchParams = useSearchParams();
    const router = useRouter();
    const dispatch = useDispatch();
    const handleCityChange = useCallback((cityId) => {
        dispatch(setSelectedCityId(cityId)); // تحديث Redux store
        onLocationChange?.({ cityId, branchId: selectedBranchId });
    }, [onLocationChange, selectedBranchId, dispatch]);

    const handleBranchChange = useCallback((branchId) => {
        dispatch(setSelectedBranchId(branchId)); // تحديث Redux store
        onLocationChange?.({ cityId: selectedCityId, branchId });
    }, [onLocationChange, selectedCityId, dispatch]);

    const resetLocation = useCallback(() => {
        dispatch(setSelectedCityId(null)); // إعادة تعيين المدينة في Redux store
        dispatch(setSelectedBranchId(null)); // إعادة تعيين الفرع في Redux store
        onLocationChange?.({ cityId: null, branchId: null });
    }, [onLocationChange, dispatch]);

    const handleDataChange = useCallback((locationData = null, overrideCarTypes = null) => {
        onLocationChange?.({ cityId: selectedCityId, branchId: selectedBranchId });

        // Convert date strings to proper format
        const startDate = selectedRange?.[0] || null;
        const endDate = selectedRange?.[1] || null;

        // حفظ البيانات في Redux للاستخدام في صفحة الحجز
        if (startDate) {
            dispatch(setSavedStartDate(startDate));
        }
        if (endDate) {
            dispatch(setSavedEndDate(endDate));
        }
        if (startTime) {
            dispatch(setSavedStartTime(startTime));
        }

        // حفظ بيانات موقع الاستلام في Redux إذا كانت متوفرة
        if (locationData && locationData.pickupAddress) {
            dispatch(setPickupLocation({
                address: locationData.pickupAddress,
                latitude: locationData.pickupLatitude,
                longitude: locationData.pickupLongitude
            }));
        }

        // استخدام overrideCarTypes إذا تم تمريره، وإلا استخدم selectedCarTypes من Redux
        const carTypesToUse = overrideCarTypes !== null ? overrideCarTypes : selectedCarTypes;

        // Safe logging with proper formatting
        const logData = {
            cityId: selectedCityId,
            branchId: selectedBranchId,
            startTime: startTime,
            startDate: startDate,
            endDate: endDate,
            duration: duration,
            deliveryType: deliveryType,
            selectedCarTypes: carTypesToUse,
            // إضافة بيانات الموقع إذا كانت متوفرة
            ...(locationData && {
                pickupAddress: locationData.pickupAddress,      
                pickupLatitude: locationData.pickupLatitude,
                pickupLongitude: locationData.pickupLongitude
            })
        };

        // Create a new URLSearchParams instance to modify search parameters
        const newSearchParams = new URLSearchParams(searchParams);
        
        // إضافة جميع المعاملات المطلوبة للـ API
        if (selectedCityId) {
            newSearchParams.set('cityId', selectedCityId);
        }
        if (selectedBranchId) {
            newSearchParams.set('branchId', selectedBranchId);
        }
        if (duration) {
            const datingType = duration === "daily" ? 1 : duration === "hourly" ? 2 : 3;
            newSearchParams.set('datingType', datingType);
        }
        if (startDate) {
            newSearchParams.set('startDate', startDate);
        }
        if (endDate) {
            newSearchParams.set('endDate', endDate);
        }
        
        // إضافة معاملات إضافية للـ API
        newSearchParams.set('page', '0');
        newSearchParams.set('size', '10');
        
        // إضافة نوع السيارات
        if (carTypesToUse && carTypesToUse.length > 0) {
            carTypesToUse.forEach(carType => {
                newSearchParams.append('typeIds', carType);
            });
        }
        
        // إضافة معاملات التصفية الإضافية
        if (searchParams.get('carName')) {
            newSearchParams.set('carName', searchParams.get('carName'));
        }
        if (searchParams.get('minPrice')) {
            newSearchParams.set('minPrice', searchParams.get('minPrice'));
        }
        if (searchParams.get('maxPrice')) {
            newSearchParams.set('maxPrice', searchParams.get('maxPrice'));
        }
        if (searchParams.get('sort')) {
            newSearchParams.set('sort', searchParams.get('sort'));
        }
        
        router.push(`/cars?${newSearchParams.toString()}`);



        cookieStorage.setItem('rentalData', JSON.stringify(logData));
    }, [selectedCityId, selectedBranchId, startTime, selectedRange, onLocationChange, searchParams, router, duration, deliveryType, selectedCarTypes, dispatch]);

    return {
        selectedCityId,
        selectedBranchId,
        handleCityChange,
        handleBranchChange,
        resetLocation,
        handleDataChange
    };
};

export default useLocationSelection; 