'use client';
import { URL } from '@/constants/api';
import { useCallback, useState } from 'react';

/**
 * Hook لجلب مواعيد العمل للفرع المحدد
 */
const useBranchWorkingHours = () => {
  const [workingHours, setWorkingHours] = useState(null);
  const [isLoading, setIsLoading] = useState(false);
  const [error, setError] = useState(null);

  /**
   * جلب مواعيد العمل للفرع
   * @param {number} branchId - معرف الفرع
   */
  const fetchWorkingHours = useCallback(async (branchId) => {
    if (!branchId) {
      setWorkingHours(null);
      return;
    }

    setIsLoading(true);
    setError(null);

    try {
      const url = URL(`/workinghours/find/branch/${branchId}/current-working-hours`);
      const response = await fetch(url);

      if (!response.ok) {
        throw new Error(`HTTP error! status: ${response.status}`);
      }

      const result = await response.json();

      if (result.status && result.data) {
        setWorkingHours(result.data);
      } else {
        throw new Error(result.message || 'فشل في جلب مواعيد العمل');
      }
    } catch (err) {
      // Error fetching working hours
      setError(err.message);
      setWorkingHours(null);
    } finally {
      setIsLoading(false);
    }
  }, []);

  /**
   * الحصول على مواعيد العمل ليوم محدد
   * @param {string} dayName - اسم اليوم (sun, mon, tues, wed, thurs, fri, sat)
   * @returns {object} - مواعيد العمل لليوم
   */
  const getDayWorkingHours = useCallback((dayName) => {
    if (!workingHours) return null;

    const dayKey = dayName.toLowerCase();
    const isOpen = workingHours[dayKey] > 0;

    if (!isOpen) {
      return {
        isOpen: false,
        openTime: null,
        closeTime: null
      };
    }

    const openHour = workingHours[`${dayKey}OpenHour`];
    const openMin = workingHours[`${dayKey}OpenMin`];
    const closeHour = workingHours[`${dayKey}CloseHour`];
    const closeMin = workingHours[`${dayKey}CloseMin`];

    return {
      isOpen: true,
      openTime: {
        hour: openHour,
        minute: openMin
      },
      closeTime: {
        hour: closeHour,
        minute: closeMin
      }
    };
  }, [workingHours]);

  /**
   * التحقق من أن الوقت داخل ساعات العمل
   * @param {number} hour  
   * @param {number} minute   
   * @param {string} dayName  
   * @returns {boolean} - هل الوقت متاح للحجز
   */
  const isTimeAvailable = useCallback((hour, minute, dayName) => {
    const dayHours = getDayWorkingHours(dayName);

    if (!dayHours || !dayHours.isOpen) {
      return false;
    }

    const { openTime, closeTime } = dayHours;
    const currentTimeInMinutes = hour * 60 + minute;
    const openTimeInMinutes = openTime.hour * 60 + openTime.minute;
    const closeTimeInMinutes = closeTime.hour * 60 + closeTime.minute;

    return currentTimeInMinutes >= openTimeInMinutes && currentTimeInMinutes < closeTimeInMinutes;
  }, [getDayWorkingHours]);

  return {
    workingHours,
    isLoading,
    error,
    fetchWorkingHours,
    getDayWorkingHours,
    isTimeAvailable
  };
};

export default useBranchWorkingHours;
