'use client';

import { useRouter } from '@/navigation';
import { setBookingLimData } from '@/store/features/lim/limousineSlice';
import cookieStorage from '@/utils/cookieStorage';
import {
    buildBookingPayload,
    buildOrderQueryParams,
    cleanupBookingData,
    getApiEndpoint,
    validateBookingData
} from '@/utils/limousineUtils';
import { message } from 'antd';
import { useState } from 'react';
import { useDispatch } from 'react-redux';
import { URL } from '../../constants/api';

/**
 * Custom Hook لإدارة حجز الليموزين
 * يدعم نوعين من الحجز: المطار وأماكن أخرى
 */
export const useLimousineBooking = () => {
  const dispatch = useDispatch();
  const router = useRouter();
  const [loading, setLoading] = useState(false);

  /**
   * إرسال طلب الحجز
   */
  const submitBooking = async (bookingData) => {
    setLoading(true);

    try {
      // التحقق من وجود token
      const token = typeof window !== 'undefined' ? cookieStorage.getItem('token') : null;
      if (!token) {
        message.error('رمز المصادقة مفقود');
        return { success: false, error: 'رمز المصادقة مفقود' };
      }

      // التأكد من وجود reservationType
      let finalBookingData = { ...bookingData };
      
      if (!finalBookingData.reservationType) {
        // تحديد نوع الحجز بناءً على البيانات المتاحة
        if (finalBookingData.toLatitude && finalBookingData.toLongitude) {
          finalBookingData.reservationType = "OTHER_PLACES";
        } else {
          finalBookingData.reservationType = "AIRPORT";
        }
      }

      // التحقق من صحة بيانات الحجز
      const validation = validateBookingData(finalBookingData);
      if (!validation.isValid) {
        message.error(validation.error);
        return { success: false, error: validation.error };
      }

      // بناء البيانات و API endpoint
      const payload = buildBookingPayload(finalBookingData);
      
      // تحديد API endpoint بناءً على نوع الحجز
      let apiEndpoint;
      if (finalBookingData.reservationType === "HOURLY") {
        apiEndpoint = URL('/limousine-reservations/hourly');
      } else {
        apiEndpoint = getApiEndpoint(finalBookingData.reservationType);
      }

      // إرسال الطلب
      const response = await fetch(apiEndpoint, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Authorization': `Bearer ${token}`
        },
        body: JSON.stringify(payload)
      });

      // معالجة الاستجابة
      const text = await response.text();
      let result;

      try {
        result = JSON.parse(text);
      } catch (parseErr) {
        // JSON parse error
        message.error('فشل في تحليل استجابة الخادم');
        return { success: false, error: 'فشل في تحليل استجابة الخادم' };
      }

      // التحقق من نجاح العملية
      if (response.ok) {
        message.success('تم الحجز بنجاح');
        
        // إعادة توجيه إلى صفحة الطلب
        const queryParams = buildOrderQueryParams(result);

        // تنظيف البيانات المحفوظة
        dispatch(setBookingLimData(null));
        cleanupBookingData();

        // الانتقال إلى صفحة الطلب
        router.push(`/order?${queryParams.toString()}`);

        return { success: true, data: result };
      } else {
        const errorMessage = result.message || 'حدث خطأ أثناء إرسال البيانات';
        message.error(errorMessage);
        return { success: false, error: errorMessage };
      }

    } catch (err) {
      // Request error
      message.error('حدث خطأ غير متوقع أثناء إرسال البيانات');
      return { success: false, error: 'حدث خطأ غير متوقع أثناء إرسال البيانات' };
    } finally {
      setLoading(false);
    }
  };

  return {
    loading,
    submitBooking
  };
}; 
