# دليل استخدام مسح sessionStorage عند Refresh
# Clear SessionStorage on Refresh Guide

## نظرة عامة | Overview

هذا الملف يوثق كيفية استخدام الـ hook الخاص بمسح sessionStorage تلقائياً عند عمل refresh للصفحة.

This file documents how to use the hook for automatically clearing sessionStorage on page refresh.

---

## المكونات | Components

### 1. `useClearOnRefresh` Hook

Hook يقوم بمسح sessionStorage تلقائياً عند اكتشاف refresh للصفحة.

**الموقع | Location:** `src/hook/useClearOnRefresh.js`

#### الباراميترات | Parameters

- `enabled` (boolean, default: `true`) - تفعيل أو تعطيل المسح التلقائي
- `excludeKeys` (string[], default: `[]`) - مفاتيح معينة لا يتم مسحها

#### مثال الاستخدام | Usage Example

```javascript
import { useClearOnRefresh } from '@/hook/useClearOnRefresh';

function MyComponent() {
  // مسح كل sessionStorage عند refresh
  useClearOnRefresh();
  
  // أو مع استثناء بعض المفاتيح
  useClearOnRefresh(true, ['keep_this_key', 'and_this_key']);
  
  return <div>محتوى المكون</div>;
}
```

---

### 2. `SessionStorageManager` Component

Component عام يطبق المسح التلقائي على كل التطبيق.

**الموقع | Location:** `src/app/(components)/SessionStorageManager.jsx`

تم إضافته في `layout.jsx` ليعمل على كل الصفحات.

---

## كيف يعمل | How It Works

### آلية الكشف عن Refresh

يستخدم `performance.getEntriesByType('navigation')` للكشف عن نوع التصفح:

- `'reload'` - يعني refresh للصفحة → يتم المسح
- `'navigate'` - يعني انتقال عادي → لا يتم المسح
- `'back_forward'` - يعني زر الرجوع/التقدم → لا يتم المسح

### خطوات العمل

1. عند تحميل الصفحة، يتحقق Hook من نوع التصفح
2. إذا كان `reload`، يتم مسح sessionStorage
3. إذا كانت هناك مفاتيح مستثناة، يتم:
   - حفظها مؤقتاً
   - مسح كل sessionStorage
   - إعادة المفاتيح المستثناة

---

## الاستخدامات المختلفة | Different Use Cases

### 1. مسح كل sessionStorage (الوضع الافتراضي)

```javascript
// في أي component
import { useClearOnRefresh } from '@/hook/useClearOnRefresh';

function MyComponent() {
  useClearOnRefresh();
  // كل sessionStorage سيتم مسحه عند refresh
}
```

### 2. استثناء مفاتيح معينة من المسح

```javascript
import { useClearOnRefresh } from '@/hook/useClearOnRefresh';

function MyComponent() {
  useClearOnRefresh(true, [
    'user_preferences',
    'theme_settings',
    'language'
  ]);
  // كل المفاتيح ستمسح ماعدا المفاتيح المذكورة
}
```

### 3. تعطيل المسح التلقائي

```javascript
import { useClearOnRefresh } from '@/hook/useClearOnRefresh';

function MyComponent() {
  const [shouldClear, setShouldClear] = useState(false);
  
  useClearOnRefresh(shouldClear);
  // المسح سيتم فقط إذا كان shouldClear = true
}
```

### 4. مسح يدوي

```javascript
import { clearSessionStorage } from '@/hook/useClearOnRefresh';

function MyComponent() {
  const handleClearManually = () => {
    // مسح كل شيء
    clearSessionStorage();
    
    // أو مع استثناء مفاتيح
    clearSessionStorage(['keep_this']);
  };
  
  return (
    <button onClick={handleClearManually}>
      مسح SessionStorage
    </button>
  );
}
```

---

## التكامل مع Hooks الأخرى | Integration with Other Hooks

### مع useFormSessionStorage

```javascript
import { useClearOnRefresh } from '@/hook/useClearOnRefresh';
import useFormSessionStorage from '@/hook/useFormSessionStorage';

function MyForm() {
  const [form] = Form.useForm();
  
  // استخدام useFormSessionStorage للنموذج
  const { saveToSession } = useFormSessionStorage('myForm', form);
  
  // مسح كل sessionStorage عند refresh
  // سيتم مسح بيانات النموذج أيضاً
  useClearOnRefresh();
  
  // أو الحفاظ على بيانات النموذج
  useClearOnRefresh(true, ['myForm']);
}
```

---

## ملاحظات مهمة | Important Notes

### ⚠️ الفرق بين sessionStorage و cookieStorage

- **sessionStorage**: يتم مسحه تلقائياً عند إغلاق التبويب (لكن ليس عند refresh)
- **cookieStorage**: يبقى موجوداً حتى انتهاء صلاحيته

هذا الـ Hook يعمل فقط على `sessionStorage` وليس `cookieStorage`.

### 📝 متى يتم المسح

- ✅ عند الضغط على F5
- ✅ عند الضغط على Ctrl+R / Cmd+R
- ✅ عند الضغط على زر Reload
- ❌ لا يتم المسح عند الانتقال بين الصفحات
- ❌ لا يتم المسح عند استخدام زر Back/Forward

### 🔧 Debugging

لتتبع عملية المسح:

```javascript
import { useClearOnRefresh } from '@/hook/useClearOnRefresh';

function MyComponent() {
  useClearOnRefresh();
  
  useEffect(() => {
    console.log('SessionStorage keys:', Object.keys(sessionStorage));
  }, []);
}
```

---

## أمثلة عملية | Practical Examples

### مثال 1: صفحة حجز Limousine

```javascript
// في LimousineForm.jsx
import { useClearOnRefresh } from '@/hook/useClearOnRefresh';

function LimousineForm() {
  // مسح نوع الإيجار عند refresh
  // لكن الاحتفاظ ببيانات المستخدم
  useClearOnRefresh(true, ['user_token', 'user_info']);
  
  // باقي الكود...
}
```

### مثال 2: الاحتفاظ بإعدادات اللغة

```javascript
import { useClearOnRefresh } from '@/hook/useClearOnRefresh';

function LanguageSettings() {
  useClearOnRefresh(true, ['selected_language', 'locale']);
}
```

### مثال 3: صفحة دفع

```javascript
import { useClearOnRefresh } from '@/hook/useClearOnRefresh';

function PaymentPage() {
  // عدم مسح sessionStorage في صفحة الدفع
  // لتجنب فقدان البيانات عند refresh
  useClearOnRefresh(false);
}
```

---

## الإعداد الحالي | Current Setup

حالياً، تم إضافة `SessionStorageManager` في:

**src/app/[locale]/layout.jsx**

```javascript
<SessionStorageManager />
```

هذا يعني أن sessionStorage سيتم مسحه تلقائياً على **كل الصفحات** عند refresh.

### لتخصيص السلوك لصفحة معينة

إذا أردت سلوكاً مختلفاً لصفحة معينة، استخدم الـ Hook مباشرة في تلك الصفحة:

```javascript
// في أي صفحة
import { useClearOnRefresh } from '@/hook/useClearOnRefresh';

function SpecialPage() {
  // تجاوز السلوك الافتراضي
  useClearOnRefresh(true, ['special_data']);
}
```

---

## الدعم والتوافق | Support & Compatibility

- ✅ يعمل في المتصفحات الحديثة (Chrome, Firefox, Safari, Edge)
- ✅ يدعم Next.js 13+ (App Router)
- ✅ يعمل مع client components فقط
- ❌ لا يعمل في server components

---

## الخلاصة | Summary

| الميزة | الوصف |
|--------|-------|
| المسح التلقائي | يتم تلقائياً عند refresh |
| الاستثناءات | يمكن استثناء مفاتيح معينة |
| المسح اليدوي | متوفر عبر دالة `clearSessionStorage` |
| التكامل | يعمل مع جميع الـ hooks الأخرى |
| الأداء | لا يؤثر على أداء التطبيق |

---

**آخر تحديث:** 2025
**الحالة:** نشط ✅

