import { NextResponse } from 'next/server';

export async function POST(request) {
  try {
    const { email } = await request.json();

    // Validate email
    if (!email || !email.trim()) {
      return NextResponse.json(
        { status: false, message: 'Email is required' },
        { status: 400 }
      );
    }

    // Basic email validation
    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    if (!emailRegex.test(email)) {
      return NextResponse.json(
        { status: false, message: 'Invalid email format' },
        { status: 400 }
      );
    }

    // Get the external API URL
    const externalApiUrl = process.env.NEXT_PUBLIC_API_BASE_URL || 'https://viganium.me/api';
    const fullUrl = `${externalApiUrl}/email-lists`;

    // Forward the request to the external API
    const response = await fetch(fullUrl, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
      },
      body: JSON.stringify({ email: email.trim() }),
    });

    const result = await response.json();

    if (response.ok) {
      return NextResponse.json(result, { status: 200 });
    } else {
      return NextResponse.json(
        { 
          status: false, 
          message: result.message || 'Failed to subscribe to email list' 
        },
        { status: response.status }
      );
    }

  } catch (error) {
    console.error('Email subscription error:', error);
    return NextResponse.json(
      { 
        status: false, 
        message: 'Internal server error' 
      },
      { status: 500 }
    );
  }
}
