import { DatePicker, Form } from 'antd';
import arEG from 'antd/es/date-picker/locale/ar_EG';
import enUS from 'antd/es/date-picker/locale/en_US';
import dayjs from 'dayjs';
import { useLocale, useTranslations } from 'next-intl';
const CustomDatePicker = ({
  name,
  label,
  placeholder = 'اختر التاريخ',
  size = 'large',
  format = 'DD-MM-YYYY',
  showTime = false, 
  defaultValue = null,
  required = true,
  disablePastDates = true, // <-- اختياري للتحكم
  maxDaysFromNow = null,
  customRules = [],
  ...props
}) => {
  const locale = useLocale();
  const localeCode = locale == 'ar' ? arEG : enUS;
  const t = useTranslations();
  //❗ دالة لتحويل الساعات إلى أيام وتعطيل التواريخ
  const createMaxDaysDisabledDate = (hours) => {
    // التأكد من أن hours رقم صحيح
    const validHours = Number(hours) || 0;
    const maxDays = Math.ceil(validHours / 24); // تحويل الساعات إلى أيام
    const today = dayjs().startOf('day');
    const maxDate = today.add(maxDays, 'day');
    
    
    return (current) => {
      if (!current) return false;
      
      // تعطيل التواريخ السابقة
      if (disablePastDates && current < today) {
        return true;
      }
      
      // تعطيل التواريخ بعد الحد الأقصى
      if (current > maxDate) {
        return true;
      }
      
      return false;
    };
  };

  //❗ دالة افتراضية لتعطيل التواريخ السابقة فقط
  const defaultDisabledDate = (current) => {
    if (!disablePastDates) return false;
    return current && current < dayjs().startOf('day');
  };

  //❗ دالة لتعطيل الأوقات السابقة في اليوم الحالي
  const disabledTime = (current) => {
    if (!current || !showTime) return {};
    
    const now = dayjs();
    const isToday = current.isSame(now, 'day');
    
    if (!isToday) return {};
    
    const currentHour = now.hour();
    const currentMinute = now.minute();
    
    return {
      disabledHours: () => {
        const hours = [];
        for (let i = 0; i < currentHour; i++) {
          hours.push(i);
        }
        return hours;
      },
      disabledMinutes: (selectedHour) => {
        if (selectedHour === currentHour) {
          const minutes = [];
          for (let i = 0; i < currentMinute; i++) {
            minutes.push(i);
          }
          return minutes;
        }
        return [];
      }
    };
  };

  // تحديد الدالة المناسبة للتعطيل
  const finalDisabledDate = maxDaysFromNow && Number(maxDaysFromNow) > 0
    ? createMaxDaysDisabledDate(maxDaysFromNow) 
    : defaultDisabledDate;

  return (
    <Form.Item
      name={name}
      // label={label}
      rules={customRules.length > 0 ? customRules : [{ required: required, message: ` ${t('required')}` }]}
    >
      <DatePicker
        placeholder={placeholder}
        size={size}
        format={format}
        showTime={showTime ? { 
          minuteStep: 30,
          format: 'HH:mm',
          placeholder: {
            hour: 'ساعة',
            minute: 'دقيقة'
          },
          showNow: false,
          hideDisabledOptions: true
        } : false}
        locale={localeCode}
        defaultValue={defaultValue ? dayjs(defaultValue) : null}
        disabledDate={finalDisabledDate}
        disabledTime={disabledTime}
        style={{
          width: '100%',
          direction: locale === 'ar' ? 'rtl' : 'ltr',
          textAlign: locale === 'ar' ? 'right' : 'left',
        }}
        className={props.className}
        {...props}
      />
    </Form.Item>
  );
};

export default CustomDatePicker;
