'use client';

import {
  GoogleMap,
  InfoWindow,
  Marker,
  useJsApiLoader,
} from "@react-google-maps/api";
import { Button } from "antd";
import {
  ClockCircleOutlined,
  EnvironmentOutlined,
  PhoneOutlined,
  MessageOutlined,
} from "@ant-design/icons";
import { useLocale } from "next-intl";
import { useState } from "react";

export default function BranchesMap({ branches = [], selectedBranch = null }) {
  const locale = useLocale();
  const [selectedMarker, setSelectedMarker] = useState(null);

  const { isLoaded } = useJsApiLoader({
    googleMapsApiKey: "AIzaSyAtUOb461InzoQoGEVKKVqqLf2NbwSjqdk",
    libraries: ["places"],
  });

  // حساب المركز المتوسط للخريطة
  const getMapCenter = () => {
    // إذا كان هناك فرع محدد، استخدم إحداثياته
    if (selectedBranch && selectedBranch.latitude && selectedBranch.longitude) {
      return { lat: selectedBranch.latitude, lng: selectedBranch.longitude };
    }

    if (branches.length === 0) {
      return { lat: 21.4858, lng: 39.1925 }; // جدة كمركز افتراضي
    }

    const validBranches = branches.filter(branch =>
      branch.latitude && branch.longitude
    );

    if (validBranches.length === 0) {
      return { lat: 21.4858, lng: 39.1925 };
    }

    const centerLat = validBranches.reduce((sum, branch) => sum + branch.latitude, 0) / validBranches.length;
    const centerLng = validBranches.reduce((sum, branch) => sum + branch.longitude, 0) / validBranches.length;

    return { lat: centerLat, lng: centerLng };
  };

  const handleMarkerClick = (branch) => {
    // عرض معلومات الفرع في InfoWindow
    setSelectedMarker(branch);
  };

  const handleMapClick = () => {
    setSelectedMarker(null);
  };

  if (!isLoaded) {
    return <div className="w-full h-full bg-gray-200 flex items-center justify-center">جارٍ تحميل الخريطة...</div>;
  }

  const mapCenter = getMapCenter();

  // تحديد الفروع التي سيتم عرضها
  const branchesToShow = selectedBranch ? [selectedBranch] : branches;

  // تحديد مستوى التكبير
  const zoomLevel = selectedBranch ? 16 : (branches.length > 1 ? 8 : 16);

  return (
    <div className="w-full h-full relative">
      <GoogleMap
        center={mapCenter}
        zoom={zoomLevel}
        mapContainerStyle={{ width: "100%", height: "100%" }}
        onClick={handleMapClick}
        options={{
          mapTypeControl: false,
          fullscreenControl: false,
          streetViewControl: false,
        }}
      >
        {branchesToShow.map((branch, index) => {
          if (!branch.latitude || !branch.longitude) return null;

          // التحقق من أن هذا الفرع هو المحدد
          const isSelected = selectedMarker && (
            (selectedMarker.branchId && branch.branchId && selectedMarker.branchId === branch.branchId) ||
            (selectedMarker.id && branch.id && selectedMarker.id === branch.id) ||
            selectedMarker === branch
          );

          return (
            <Marker
              key={branch.branchId || branch.id || index}
              position={{ lat: branch.latitude, lng: branch.longitude }}
              onClick={() => handleMarkerClick(branch)}
              title={locale === 'en' ? branch.branchName : branch.branchArName}
            >
              {isSelected && (
                <InfoWindow
                  onCloseClick={() => setSelectedMarker(null)}
                >
                  <div className="p-2 max-w-xs min-w-[280px] font-cairo" dir={locale === 'ar' ? 'rtl' : 'ltr'} style={{ fontFamily: 'var(--font-cairo, Cairo, sans-serif)' }}>
                    {/* العنوان */}
                    <h3 className="font-bold text-base mb-1 text-gray-900 border-b pb-1 border-gray-200">
                      {locale === 'en' ? branch.branchName : branch.branchArName}
                    </h3>
                    
                    {/* معلومات الفرع */}
                    <div className="space-y-1">
                      {/* العنوان */}
                      {branch.addressArabic && (
                        <div className="flex items-start gap-1.5 text-sm">
                          <EnvironmentOutlined className="text-main flex-shrink-0" style={{ fontSize: '14px', marginTop: '2px' }} />
                          <div className="flex-1">
                            <span className="font-semibold text-gray-700 block">
                              {locale === 'en' ? 'Address' : 'العنوان'}
                            </span>
                            <span className="text-gray-600">
                              {branch.addressArabic}
                            </span>
                          </div>
                        </div>
                      )}
                      {branch.addressEnglish && locale === 'en' && (
                        <div className="flex items-start gap-1.5 text-sm">
                          <EnvironmentOutlined className="text-main flex-shrink-0" style={{ fontSize: '14px', marginTop: '2px' }} />
                          <div className="flex-1">
                            <span className="font-semibold text-gray-700 block">Address</span>
                            <span className="text-gray-600">
                              {branch.addressEnglish}
                            </span>
                          </div>
                        </div>
                      )}
                      
                      {/* الهاتف */}
                      {branch.mobile && (
                        <div className="flex items-start gap-1.5 text-sm">
                          <MessageOutlined className="text-main flex-shrink-0" style={{ fontSize: '14px', marginTop: '2px' }} />
                          <div className="flex-1">
                            <span className="font-semibold text-gray-700 block">
                              {locale === 'en' ? 'Phone' : 'الهاتف'}
                            </span>
                            <a 
                              href={`tel:${branch.mobile.replace(/[^0-9+]/g, '')}`}
                              className="text-main hover:text-main/80 transition-colors"
                              dir="ltr"
                            >
                              {branch.mobile}
                            </a>
                          </div>
                        </div>
                      )}
                      
                      {/* الخط الساخن */}
                      {branch.phone1 && (
                        <div className="flex items-start gap-1.5 text-sm">
                          <PhoneOutlined className="text-main flex-shrink-0" style={{ fontSize: '14px', marginTop: '2px' }} />
                          <div className="flex-1">
                            <span className="font-semibold text-gray-700 block">
                              {locale === 'en' ? 'Hotline' : 'الخط الساخن'}
                            </span>
                            <a 
                              href={`tel:${branch.phone1.replace(/[^0-9+]/g, '')}`}
                              className="text-gray-600 hover:text-main transition-colors"
                            >
                              {branch.phone1}
                            </a>
                          </div>
                        </div>
                      )}
                      
                      {/* ساعات العمل */}
                      {branch.workingHoures && (
                        <div className="flex items-start gap-1.5 text-sm">
                          <ClockCircleOutlined className="text-main flex-shrink-0" style={{ fontSize: '14px', marginTop: '2px' }} />
                          <div className="flex-1">
                            <span className="font-semibold text-gray-700 block">
                              {locale === 'en' ? 'Working Hours' : 'ساعات العمل'}
                            </span>
                            <span className="text-gray-600">
                              {branch.workingHoures}
                            </span>
                          </div>
                        </div>
                      )}
                    </div>
                    
                    {/* زر فتح في Google Maps */}
                    {(branch.googlePlaceId || (branch.latitude && branch.longitude)) && (
                      <Button
                        type="primary"
                        onClick={() => {
                          if (branch.googlePlaceId) {
                            window.open(branch.googlePlaceId, '_blank', 'noopener,noreferrer');
                          } else if (branch.latitude && branch.longitude) {
                            window.open(`https://www.google.com/maps?q=${branch.latitude},${branch.longitude}`, '_blank', 'noopener,noreferrer');
                          }
                        }}
                        className="mt-1 w-full h-8 font-semibold bg-main border-main hover:bg-main/90"
                      >
                        {locale === 'en' ? 'Open in Google Maps' : 'فتح في خرائط Google'}
                      </Button>
                    )}
                  </div>
                </InfoWindow>
              )}
            </Marker>
          );
        })}
      </GoogleMap>
    </div>
  );
} 