"use client";

import { CalendarOutlined } from '@ant-design/icons';
import { ContentCard } from '@components/ui';
import { Button, Radio } from 'antd';
import { useTranslations } from 'next-intl';
import { useState } from 'react';
import './DurationSelector.css';

const DurationSelector = ({
    options = [4, 5, 6, 7, 8, 9, 10, 11, 12],
    value,
    onChange,
    title = "المدة:",
    className = "",
    showIcon = true,
    disabled = false,
    unit = "ساعات", // إضافة وحدة القياس
    showAddHoursButton = false, // إظهار زر إضافة ساعات
    onAddHours = null // دالة إضافة ساعات
}) => {
    const t = useTranslations();
    const [selectedValue, setSelectedValue] = useState(value || options[1]); // Default to second option (5 hours)

    const handleChange = (e) => {
        const newValue = e.target.value;
        setSelectedValue(newValue);
        if (onChange) {
            onChange(newValue);
        }
    };

    // دالة لتنسيق النص بناءً على الوحدة والقيمة
    const formatDurationText = (count, currentUnit) => {
        if (currentUnit === "أيام") {
            if (count === 1) return "يوم";
            if (count === 2) return "يومين";
        }
        return `${count} ${currentUnit}`;
    };

    return (
        <ContentCard className={'!p-1'}>
            {/* Header */}
            <div className="flex items-center justify-between p-4 border-b border-gray-100 bg-gray-50 rounded-t-lg">
                <div className="flex gap-2 items-center">
                    {showIcon && (
                        <CalendarOutlined className="text-gray-500 ml-2 text-lg" />
                    )}
                    <span className="font-cairo font-bold text-[18px] leading-[100%] tracking-[0] text-[#162A2B] text-right align-middle">
                        {title}
                    </span>
                </div>
                {showAddHoursButton && (
                    <Button
                        onClick={onAddHours}
                        type='primary'
                        size='small'
                        className='  !font-medium !px-2  !py-2'
                    >
                        <span>+</span>
                        <span>{t('add_additional_hours')}</span>    
                    </Button>
                )}
            </div>

            {/* Options List */}
            <div className="h-80 overflow-y-auto custom-scrollbar px-[16px]">
                <Radio.Group
                    value={selectedValue}
                    onChange={handleChange}
                    className="w-full"
                    disabled={disabled}
                >
                    {options.map((hours, index) => (
                        <div
                            key={hours}
                            className={`group flex items-center gap-2 p-3  last:rounded-b-lg cursor-pointer transition-all duration-200 
                                 ${selectedValue == hours ? '!bg-[#FBF7ED] !text-main text-[18px] !border-secondary border-solid  rounded-md  border ' : 'hover:bg-gray-50'} 
                                 ${disabled ? 'opacity-50 cursor-not-allowed' : ''}`}
                            onClick={() => !disabled && handleChange({ target: { value: hours } })}
                        >
                            <Radio
                                value={hours}
                                className="duration-radio"
                                disabled={disabled}
                            />
                            {/* Duration Text */}
                            <span className={`font-medium text-base transition-colors ${selectedValue === hours
                                ? 'text-amber-800'
                                : 'text-gray-800 group-hover:text-gray-900'
                                }`}>
                                {formatDurationText(hours, unit)}
                            </span>

                            {/* Radio Button */}
                        </div>
                    ))}
                </Radio.Group>
            </div>
        </ContentCard>

    );
};

export default DurationSelector;
