'use client';

import { useState, useEffect, useRef } from 'react';
import Image from 'next/image';
import { createIntersectionObserver, shouldLoadHighQuality } from '@/utils/performanceUtils';

/**
 * مكون صورة محسّن
 * يستخدم Intersection Observer و lazy loading ذكي
 * 
 * @example
 * <OptimizedImage 
 *   src="/path/to/image.jpg"
 *   alt="وصف الصورة"
 *   width={800}
 *   height={600}
 *   priority={false}
 * />
 */
const OptimizedImage = ({
  src,
  alt,
  width,
  height,
  priority = false,
  quality,
  className = '',
  fill = false,
  sizes,
  objectFit = 'cover',
  placeholder = 'blur',
  blurDataURL,
  onLoad,
  onError,
  ...props
}) => {
  const [isVisible, setIsVisible] = useState(priority);
  const [isLoaded, setIsLoaded] = useState(false);
  const [hasError, setHasError] = useState(false);
  const imgRef = useRef(null);
  const highQuality = shouldLoadHighQuality();

  // تحديد الجودة بناءً على سرعة الاتصال إذا لم تكن محددة
  const imageQuality = quality || (highQuality ? 85 : 70);

  useEffect(() => {
    // إذا كانت الصورة ذات أولوية، لا نحتاج لـ Intersection Observer
    if (priority) return;

    const observer = createIntersectionObserver(
      (entries) => {
        entries.forEach((entry) => {
          if (entry.isIntersecting) {
            setIsVisible(true);
            observer?.disconnect();
          }
        });
      },
      {
        threshold: 0.01,
        rootMargin: '100px', // تحميل الصورة قبل 100px من دخولها viewport
      }
    );

    if (imgRef.current && observer) {
      observer.observe(imgRef.current);
    }

    return () => {
      if (observer) {
        observer.disconnect();
      }
    };
  }, [priority]);

  const handleLoad = (e) => {
    setIsLoaded(true);
    if (onLoad) onLoad(e);
  };

  const handleError = (e) => {
    setHasError(true);
    if (onError) onError(e);
  };

  // إذا كانت الصورة غير مرئية بعد، عرض placeholder
  if (!isVisible) {
    return (
      <div
        ref={imgRef}
        className={`bg-gray-200 animate-pulse ${className}`}
        style={{
          width: fill ? '100%' : width,
          height: fill ? '100%' : height,
          position: fill ? 'relative' : 'static',
        }}
      />
    );
  }

  // إذا كان هناك خطأ في تحميل الصورة
  if (hasError) {
    return (
      <div
        className={`bg-gray-300 flex items-center justify-center ${className}`}
        style={{
          width: fill ? '100%' : width,
          height: fill ? '100%' : height,
          position: fill ? 'relative' : 'static',
        }}
      >
        <svg
          className="w-12 h-12 text-gray-400"
          fill="none"
          stroke="currentColor"
          viewBox="0 0 24 24"
        >
          <path
            strokeLinecap="round"
            strokeLinejoin="round"
            strokeWidth={2}
            d="M4 16l4.586-4.586a2 2 0 012.828 0L16 16m-2-2l1.586-1.586a2 2 0 012.828 0L20 14m-6-6h.01M6 20h12a2 2 0 002-2V6a2 2 0 00-2-2H6a2 2 0 00-2 2v12a2 2 0 002 2z"
          />
        </svg>
      </div>
    );
  }

  return (
    <div
      ref={imgRef}
      className={`relative overflow-hidden ${className}`}
      style={{
        width: fill ? '100%' : width,
        height: fill ? '100%' : height,
      }}
    >
      <Image
        src={src}
        alt={alt}
        width={fill ? undefined : width}
        height={fill ? undefined : height}
        fill={fill}
        quality={imageQuality}
        loading={priority ? 'eager' : 'lazy'}
        priority={priority}
        sizes={sizes}
        className={`
          ${fill ? 'object-' + objectFit : ''}
          ${isLoaded ? 'opacity-100' : 'opacity-0'}
          transition-opacity duration-300
        `}
        onLoad={handleLoad}
        onError={handleError}
        placeholder={blurDataURL ? placeholder : 'empty'}
        blurDataURL={blurDataURL}
        {...props}
      />
      
      {/* Skeleton أثناء التحميل */}
      {!isLoaded && (
        <div className="absolute inset-0 bg-gray-200 animate-pulse" />
      )}
    </div>
  );
};

export default OptimizedImage;

