"use client";
import img from "@/constants/img";
import useIsMobile from "@/hook/useIsMobile";
import { Link, usePathname, useRouter } from "@/navigation";
import { isAuthenticated } from "@/utils/authUtils";
import { useLocale, useTranslations } from "next-intl";
import Image from "next/image";
import { useSearchParams } from "next/navigation";
import { lazy, Suspense, useEffect, useState } from "react";
import { useSelector } from "react-redux";
import PointsSidebar from "../sidebar/PonitsSidebar";
import GuestActions from "./GuestActions";
import LanguageModal from "./LanguageModal";
import "./navbar.css";
import NavLinks from "./NavLinks";
import PointsDisplay from "./PointsDisplay";
import UserDropdown from "./UserDropdown";

// Lazy load BottomNavigation only when needed
const BottomNavigation = lazy(() => import("./mobile/BottomNavigation"));

const Navbar = () => {
  const t = useTranslations();
  const { user } = useSelector((state) => state.profile);
  const pathname = usePathname();
  const router = useRouter();
  const locale = useLocale();
  const searchParams = useSearchParams();

  // State management
  const [isLangModalVisible, setLangModalVisible] = useState(false);
  const [isMenuOpen, setIsMenuOpen] = useState(false);
  const [isScrolled, setIsScrolled] = useState(false);
  const [pointsOpen, setPointsOpen] = useState(false);
  const [isLoggedIn, setIsLoggedIn] = useState(false);

  // Check if screen is mobile or tablet size using custom hook
  const { isMobile, isTablet } = useIsMobile(1000);

  // Effects
  useEffect(() => {
    document.documentElement.dir = locale === "en" ? "ltr" : "rtl";
  }, [locale]);

  useEffect(() => {
    const handleScroll = () => {
      setIsScrolled(window.scrollY > 50);
    };
    window.addEventListener("scroll", handleScroll);
    return () => window.removeEventListener("scroll", handleScroll);
  }, []);

  useEffect(() => {
    const checkAuthStatus = () => {
      setIsLoggedIn(isAuthenticated());
    };

    // Check on mount
    checkAuthStatus();

    // Check on pathname change (when user navigates)
    checkAuthStatus();
  }, [pathname]);

  // Also check when component becomes visible (after login/logout)
  useEffect(() => {
    const handleVisibilityChange = () => {
      if (document.visibilityState === 'visible') {
        setIsLoggedIn(isAuthenticated());
      }
    };

    document.addEventListener('visibilitychange', handleVisibilityChange);
    return () => document.removeEventListener('visibilitychange', handleVisibilityChange);
  }, []);

  // Handlers
  const changeLanguage = async (newLocale) => {
    // Check if current path is a blog page with slug (e.g., /blogs/45/slug)
    const blogMatch = pathname.match(/^\/blogs\/(\d+)\/([^\/]+)/);
    
    if (blogMatch) {
      const blogId = blogMatch[1];
      
      try {
        // Fetch blog data to get the correct slug for the new locale
        const res = await fetch(`https://viganium.me/api/blog/find/${blogId}`, {
          headers: { 'Content-Type': 'application/json', 'Accept': 'application/json' }
        });
        
        if (res.ok) {
          const response = await res.json();
          const blogData = response?.status && response?.data ? response.data : response;
          
          if (blogData) {
            // Get the correct slug for the new locale
            const newSlug = newLocale === 'ar' 
              ? (blogData.slugAr || 'blog-post')
              : (blogData.slug || 'blog-post');
            
            // Build new path with correct slug
            const newPath = `/blogs/${blogId}/${newSlug}`;
            const params = new URLSearchParams(searchParams.toString());
            const queryString = params.toString();
            const pathnameWithParams = queryString ? `${newPath}?${queryString}` : newPath;
            
            router.replace(pathnameWithParams, { locale: newLocale });
            setLangModalVisible(false);
            return;
          }
        }
      } catch (error) {
        console.error('Error fetching blog data for language switch:', error);
        // Fall through to default behavior
      }
    }
    
    // Default behavior for non-blog pages or if blog fetch fails
    const params = new URLSearchParams(searchParams.toString());
    const queryString = params.toString();
    const pathnameWithParams = queryString ? `${pathname}?${queryString}` : pathname;

    router.replace(pathnameWithParams, { locale: newLocale });
    setLangModalVisible(false);
  };

  const showLangModal = () => setLangModalVisible(true);
  const closeLangModal = () => setLangModalVisible(false);
  // Helper function to determine navbar background
  const getNavbarBackground = () => {
    if (isScrolled) return 'bg-main';

    const whiteBackgroundPaths = [
      '/membership', '/branches', '/thanks-page', '/blogs', '/profile',
      '/current-reservations', '/current-limousine-reservations', '/invite',
      'order', '/cars/', '/blogs/', '/terms-of-use', '/privacy-policy',
      '/limo-checkout', '/limousine-details', '/limo/', '/booking', '/payment-result' , '/about'
    ];

    const hasWhiteBackground = whiteBackgroundPaths.some(path =>
      pathname === path || pathname.includes(path)
    );

    return hasWhiteBackground ? 'bg-main' : 'bg-main1';
  };

  return (
    <div className={`navbar-container relative ${locale === "ar" ? "rtl" : "ltr"}`}>
      <div
        className={`navbar-main max-lg:!bg-main fixed py-2 px-8 top-0 right-0 left-0 ${getNavbarBackground()}`}
        style={{ transition: 'all 0.6s ease-in-out' }}
      >


        {/* Main Navbar Content */}
        <div className="flex items-center justify-between w-full">
          {/* Logo Section */}
          <div className="flex items-center gap-4">
            <Link href="/">
              <Image src={img.logo} alt="الغزال" className="max-sm:p-3" priority width={90} height={60} />
            </Link>
          </div>
          {!(isMobile || isTablet) && (
            <>
              {/* Navigation Links */}
              < NavLinks isMenuOpen={isMenuOpen} />
            </>
          )}


          {/* Top Bar - Desktop and Tablet */}
          <div className={`top-bar ${isMobile ? 'max-sm:!hidden' : ''}`}>
            {isLoggedIn ? (
              <>
                <PointsDisplay onPointsClick={() => setPointsOpen(true)} />
                {!(isMobile || isTablet) && (
                  <div className="flex items-center gap-4 text-white">
                    <UserDropdown onLanguageClick={showLangModal} />
                  </div>
                )}

              </>
            ) : (
              <GuestActions onLanguageChange={changeLanguage} />
            )}
          </div>

          {/* Mobile Points Display */}
          {isLoggedIn && isMobile && (
            <PointsDisplay onPointsClick={() => setPointsOpen(true)} isMobile />
          )}
        </div>

        {/* Hamburger Menu for Tablet */}
        {isTablet && (
          <button
            className="hamburger-menu tablet:hidden"
            onClick={() => setIsMenuOpen(!isMenuOpen)}
            aria-label="Toggle menu"
          >
            <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2">
              {isMenuOpen ? (
                <path d="M6 18L18 6M6 6l12 12" />
              ) : (
                <>
                  <path d="M3 12h18M3 6h18M3 18h18" />
                </>
              )}
            </svg>
          </button>
        )}


      </div>

      {/* Language Modal */}
      <LanguageModal
        isVisible={isLangModalVisible}
        onClose={closeLangModal}
        onLanguageChange={changeLanguage}
      />

      {/* Points Sidebar */}
      <PointsSidebar t={t} open={pointsOpen} onClose={() => setPointsOpen(false)} />

      {/* Bottom Navigation - Render on mobile and tablet */}
      {(isMobile || isTablet) && (
        <Suspense fallback={null}>
          <BottomNavigation onPointsClick={() => setPointsOpen(true)} />
        </Suspense>
      )}
    </div>
  );
};

export default Navbar;
