'use client';

import { formatCurrency } from '@/utils/format';
import { Button, Form } from 'antd';
import { useTranslations } from 'next-intl';
import { useCallback, useEffect, useRef, useState } from 'react';

const PromoCodeInputLimousine = ({
    isLang,
    name = "promoCode",
    form,
    validateReservation,
    dir,
    promoDiscount = 0 // قيمة الخصم من البرومو كود
}) => {
    const t = useTranslations('points_sidebar');
    const tGeneral = useTranslations();
    const validationTimeoutRef = useRef(null);
    const loadingTimeoutRef = useRef(null);
    
    // Loading state for apply button
    const [isApplying, setIsApplying] = useState(false);
    
    // Local state for input value to ensure it updates immediately
    const [inputValue, setInputValue] = useState('');

    // Sync input value with form value on mount
    useEffect(() => {
        const formValue = form?.getFieldValue(name);
        if (formValue) {
            setInputValue(formValue);
        }
    }, [form, name]);

    // Cleanup timeouts on unmount
    useEffect(() => {
        return () => {
            if (validationTimeoutRef.current) {
                clearTimeout(validationTimeoutRef.current);
            }
            if (loadingTimeoutRef.current) {
                clearTimeout(loadingTimeoutRef.current);
            }
        };
    }, []);

    // Handle promo code application with debouncing for limousine
    const handleApply = useCallback(() => {
        try {
            const value = form.getFieldValue(name);

            // Don't apply if already applying
            if (isApplying) return;

            // Start loading state
            setIsApplying(true);

            // Clear value if empty and call API
            if (!value || value.trim() === '') {
                form.setFieldValue(name, null);
            }

            // Clear previous timeouts
            if (validationTimeoutRef.current) {
                clearTimeout(validationTimeoutRef.current);
            }
            if (loadingTimeoutRef.current) {
                clearTimeout(loadingTimeoutRef.current);
            }

            // Set minimum loading time for better UX (at least 800ms)
            loadingTimeoutRef.current = setTimeout(() => {
                setIsApplying(false);
            }, 2000); // Auto-stop loading after 2 seconds max

            // Debounce validation to prevent multiple API calls
            validationTimeoutRef.current = setTimeout(async () => {
                try {
                    // Always validate reservation to recalculate limousine price
                    // This will call the limousine API endpoint: /limousine-reservations/price-preview
                    if (validateReservation) {
                        // Get current form values to include promo code
                        const formValues = form.getFieldsValue();

                        // Trigger the validation with current promo code
                        // The validateReservation function should handle the limousine-specific API call
                        await validateReservation(formValues);
                    }
                } catch (error) {
                 } finally {
                    // Clear loading timeouts and stop loading
                    if (loadingTimeoutRef.current) {
                        clearTimeout(loadingTimeoutRef.current);
                    }
                    setIsApplying(false);
                }
            }, 500); // Wait 500ms after user stops typing

        } catch (error) {
            // Error applying limousine promo code
            setIsApplying(false);
        }
    }, [form, name, validateReservation, isApplying]);

    // Handle input change with auto-validation when cleared
    const handleInputChange = useCallback((e) => {
        const value = e.target.value;

        // Update local state immediately for responsive UI
        setInputValue(value);

        // Update form field value
        form.setFieldValue(name, value || null);

        // If input is cleared, trigger validation to recalculate price
        if (!value || value.trim() === '') {
            // Clear previous timeouts
            if (validationTimeoutRef.current) {
                clearTimeout(validationTimeoutRef.current);
            }
            if (loadingTimeoutRef.current) {
                clearTimeout(loadingTimeoutRef.current);
            }

            // Start loading state for visual feedback
            setIsApplying(true);

            // Set auto-stop loading timeout
            loadingTimeoutRef.current = setTimeout(() => {
                setIsApplying(false);
            }, 2000);

            // Debounce validation to prevent multiple API calls
            validationTimeoutRef.current = setTimeout(async () => {
                try {
                    // Trigger validation with cleared promo code to recalculate limousine price
                    if (validateReservation) {
                        // Get current form values with cleared promo code
                        const formValues = form.getFieldsValue();
                        formValues[name] = null; // Ensure promo code is null

                        // Trigger the validation with cleared promo code
                        await validateReservation(formValues);
                    }
                } catch (error) {
                    // Error clearing promo code
                } finally {
                    // Clear loading timeouts and stop loading
                    if (loadingTimeoutRef.current) {
                        clearTimeout(loadingTimeoutRef.current);
                    }
                    setIsApplying(false);
                }
            }, 300); // Shorter delay for clearing (300ms)
        }
    }, [form, name, validateReservation, setIsApplying]);

    // Handle Enter key press to apply promo code
    const handleKeyPress = useCallback((e) => {
        if (e.key === 'Enter') {
            e.preventDefault();
            handleApply();
        }
    }, [handleApply]);

    // Handle clear button (if using allowClear)
    const handleClear = useCallback(() => {
        // Update local state
        setInputValue('');
        
        // Set field to null
        form.setFieldValue(name, null);
        
        // Trigger the same logic as clearing input manually
        handleInputChange({ target: { value: '' } });
    }, [form, name, handleInputChange]);

    return (
        <Form.Item name={name} noStyle>
            <div className="w-full h-[52px] bg-white rounded-[6px] p-[5px] flex justify-between items-center gap-[10px] border border-gray-200 max-md:h-[42px]">
                {/* Promo Code Input */}
                <div className="flex-1 relative flex items-center">
                    <input
                        type="text"
                        placeholder={t("promo_code")}
                        dir={dir || (isLang === "ar" ? 'rtl' : 'ltr')}
                        className={`w-full h-[42px] max-md:h-[32px] px-3 ${isLang === 'ar' ? 'pl-8' : 'pr-8'} border-0 outline-none font-[Cairo] text-[14px] ${isLang === 'ar' ? 'text-end' : 'text-start'}`}
                        value={inputValue || ''}
                        onChange={handleInputChange}
                        onKeyPress={handleKeyPress}
                        maxLength={20}
                    />
                    {/* زر الحذف */}
                    {inputValue && (
                        <button
                            type="button"
                            onClick={async () => {
                                setInputValue('');
                                form?.setFieldValue(name, null);
                                // Trigger the same logic as clearing input manually
                                handleInputChange({ target: { value: '' } });
                            }}
                            className={`absolute ${isLang === 'en' ? 'left-2' : 'right-2'} top-1/2 -translate-y-1/2 flex items-center gap-1 text-red-600 hover:text-red-500 transition-colors font-[Cairo] text-[13px] font-medium`}
                        >
                            <svg xmlns="http://www.w3.org/2000/svg" className="h-4 w-4" viewBox="0 0 20 20" fill="currentColor">
                                <path fillRule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zM8.707 7.293a1 1 0 00-1.414 1.414L8.586 10l-1.293 1.293a1 1 0 101.414 1.414L10 11.414l1.293 1.293a1 1 0 001.414-1.414L11.414 10l1.293-1.293a1 1 0 00-1.414-1.414L10 8.586 8.707 7.293z" clipRule="evenodd" />
                            </svg>
                            <span>{isLang === 'ar' ? 'حذف' : 'Clear'}</span>
                        </button>
                    )}
                </div>
                {/* Apply Button */}
                <Button
                    type="primary"
                    onClick={handleApply}
                    loading={isApplying}
                    disabled={isApplying}
                    className="shadow-none bg-gray-800 hover:bg-gray-700 h-[42px] max-md:h-[32px] max-md:text-[12px] px-6 rounded-[4px] font-[Cairo] font-medium disabled:bg-gray-400"
                >
                    {isApplying ? tGeneral('applying') : tGeneral('apply')}
                </Button>
            </div>
            
            {/* عرض قيمة الخصم */}
            {promoDiscount > 0 && (
                <div className={`mt-2 font-[Cairo] text-green-600 text-[14px] font-semibold ${isLang === 'ar' ? 'text-right' : 'text-left'}`}>
                    <span>{isLang === 'ar' ? 'تم تفعيل الكود وفرت ' : 'The code has been activated and saved: '}</span>
                    <span>{formatCurrency(promoDiscount)} {tGeneral('SAR')}</span>
                </div>
            )}
        </Form.Item>
    );
};

export default PromoCodeInputLimousine;
