import { Link } from '@/navigation';
import { Card } from 'antd';
import { format, isValid } from 'date-fns';
import { arSA, enUS } from 'date-fns/locale';
import Image from 'next/image';

const CardBlog = ({ isLang, item }) => {
    const parseDateString = (dateString) => {
        if (!dateString) return null;

        // Handle ISO date format from API (e.g., "2025-08-26T21:00:00")
        const date = new Date(dateString);
        return isValid(date) ? date : null;
    };

    const formatDate = (dateString) => {
        const parsedDate = parseDateString(dateString);
        if (parsedDate && isValid(parsedDate)) {
            return isLang === 'ar'
                ? format(parsedDate, 'dd MMMM yyyy', { locale: arSA })
                : format(parsedDate, 'MMMM dd, yyyy', { locale: enUS });
        }
        return isLang === 'ar' ? 'تاريخ غير صحيح' : 'Invalid date';
    };

    // Generate slug from title
    const generateSlug = (title) => {
        if (!title) return 'blog-post';
        return title
            .toLowerCase()
            .replace(/[^\w\s-]/g, '') // Remove special characters
            .replace(/\s+/g, '-') // Replace spaces with hyphens
            .replace(/-+/g, '-') // Replace multiple hyphens with single
            .trim('-'); // Remove leading/trailing hyphens
    };

    const blogUrl = `/blogs/${item.blogId}/${item.slug || generateSlug(item.title)}`;

    return (
        <Card
            hoverable
            cover={
                <Link href={blogUrl} className="block">
                    <Image
                        width={800}
                        height={400}
                        loading="lazy"
                        src={item.image}
                        alt={item.title}
                        className="w-full h-auto object-cover rounded-t-2xl"
                    />
                </Link>
            }
            className="rounded-2xl shadow-md h-full flex flex-col justify-between mt-3"
            styles={{
                body: {
                    padding: '1rem',
                },
            }}
        >
            <div className='px-3'>
                <div className="flex items-center gap-2 text-gray-500 text-sm mb-2">
                    <i className="pi pi-calendar" />
                    <span>{formatDate(item.publishedAt)}</span>
                </div>
                <Link href={blogUrl} className="text-main hover:text-secondary transition duration-300">
                    <h3 className="text-lg font-semibold mt-2">
                        {isLang == "en" ? item.title?.split(' ').slice(0, 7).join(' ') : item.titleAr}

                        {/* {item.title?.split(' ').slice(0, 7).join(' ')} */}
                    </h3>
                    <p className="text-gray-600 mt-1 text-sm">
                        {isLang == "en" ? item.shortText?.split(' ').slice(0, 15).join(' ') : item.shortTextAr?.split(' ').slice(0, 15).join(' ')}...
                    </p>
                </Link>
            </div>
        </Card>
    );
};

export default CardBlog;
