'use client';

import img from '@/constants/img';
import { Link } from '@/navigation';
import { Card, Skeleton } from 'antd';
import { format, isValid } from 'date-fns';
import { arSA, enUS } from 'date-fns/locale';
import { CalendarRange } from 'lucide-react';
import Image from 'next/image';
import { useState } from 'react';

const CardBlog = ({ isLang, item }) => {
    const [imageLoading, setImageLoading] = useState(true);
    const [imageError, setImageError] = useState(false);

    // Check if image exists and is not null
    const getImageSrc = () => {
        const imageSrc = isLang === 'ar' ? item.image : item.imageAr;
        return imageSrc && imageSrc !== 'null' && imageSrc !== null ? imageSrc : null;
    };

    const hasValidImage = getImageSrc() !== null;

    const parseDateString = (dateString) => {
        if (!dateString) return null;

        // Handle ISO date format from API (e.g., "2025-08-26T21:00:00")
        const date = new Date(dateString);
        return isValid(date) ? date : null;
    };

    const formatDate = (dateString) => {
        const parsedDate = parseDateString(dateString);
        if (parsedDate && isValid(parsedDate)) {
            return isLang === 'ar'
                ? format(parsedDate, 'dd MMMM yyyy', { locale: arSA })
                : format(parsedDate, 'MMMM dd, yyyy', { locale: enUS });
        }
        return isLang === 'ar' ? 'تاريخ غير صحيح' : 'Invalid date';
    };

    // Create slug-based URL for better SEO - use language-specific slug
    let slug;
    if (isLang === 'ar') {
        slug = item.slugAr || item.titleAr
            ?.replace(/\s+/g, '-')
            .replace(/[^\w\u0600-\u06FF-]/g, '-')
            .replace(/-+/g, '-')
            .toLowerCase() || item.slug || 'blog-post';
    } else {
        slug = item.slug || item.title
            ?.replace(/\s+/g, '-')
            .replace(/[^\w-]/g, '-')
            .replace(/-+/g, '-')
            .toLowerCase() || 'blog-post';
    }

    const blogUrl = `/blogs/${item.blogId}/${slug}`;

    return (
        <Card
            hoverable
            cover={
                <div className="relative w-full aspect-[3/2] overflow-hidden">
                    <Link href={blogUrl} className="block w-full h-full">
                        {hasValidImage ? (
                            <>
                                {/* Loading skeleton */}
                                {imageLoading && !imageError && (
                                    <div className="absolute inset-0 bg-gradient-to-r from-gray-200 to-gray-300 animate-pulse">
                                        <Skeleton.Image
                                            active
                                            style={{
                                                width: '100%',
                                                height: '100%',
                                                borderRadius: 0
                                            }}
                                        />
                                    </div>
                                )}

                                {/* Actual image */}
                                <Image
                                    loading="lazy"
                                    src={getImageSrc()}
                                    alt={isLang === 'ar' ? item.altImageAr : item.altImage}
                                    fill
                                    quality={85}
                                    className={`object-cover transition-all duration-300 hover:scale-105 ${imageLoading ? 'opacity-0' : 'opacity-100'
                                        }`}
                                    onLoad={() => setImageLoading(false)}
                                    onError={() => {
                                        setImageError(true);
                                        setImageLoading(false);
                                    }}
                                />

                                {/* Error state - show skeleton */}
                                {imageError && (
                                    <div className="absolute inset-0 bg-gradient-to-r from-gray-200 to-gray-300 flex items-center justify-center">
                                        <Skeleton.Image />
                                    </div>
                                )}
                            </>
                        ) : (
                            <Image
                                loading="lazy"
                                src={img.noImageBlog}
                                alt="no image blog"
                                fill
                                quality={85}
                                className={`object-cover transition-all duration-300 hover:scale-105 `}
                            />
                        )}
                    </Link>
                </div>
            }
            className={`blog-card rounded-2xl shadow-sm h-full flex flex-col transition-all duration-300 hover:shadow-lg hover:shadow-gray-200/50 hover:-translate-y-1 ${isLang === 'ar' ? 'rtl' : 'ltr'}`}
            styles={{
                body: {
                    padding: '1rem',
                    display: 'flex',
                    flexDirection: 'column',
                    height: '100%',
                },
            }}
        >
            <div className='px-3 flex flex-col h-full py-2'>
                <div className={`flex items-center gap-2 text-gray-500 text-sm mt-3 ${isLang === 'ar' ? 'text-end' : 'text-start'}`}>
                    <CalendarRange strokeWidth={1} width={16} height={16} />
                    <span>{formatDate(item.publishedAt)}</span>
                </div>
                <Link href={blogUrl} className="text-main hover:text-secondary transition duration-300 flex-1 flex flex-col">
                    <h3 className={`text-lg font-semibold mt-2 line-clamp-2 ${isLang === 'ar' ? 'text-right' : 'text-left'}`}>
                        {isLang == "en" ? item.title?.split(' ').slice(0, 7).join(' ') : item.titleAr}
                    </h3>
                    <p className={`text-gray-600 mt-1 text-sm line-clamp-3 flex-1 ${isLang === 'ar' ? 'text-right' : 'text-left'}`}>
                        {isLang == "en" ? item.shortText?.split(' ').slice(0, 15).join(' ') : item.shortTextAr?.split(' ').slice(0, 15).join(' ')}...
                    </p>
                </Link>
            </div>
        </Card>
    );
};

export default CardBlog;
