'use client';

import img from '@/constants/img';
import cookieStorage from '@/utils/cookieStorage';
import { CloseOutlined } from '@ant-design/icons';
import { useTranslations } from 'next-intl';
import Image from 'next/image';
import { useEffect, useState } from 'react';

const AppInstallBanner = () => {
  const t = useTranslations();
  const [showBanner, setShowBanner] = useState(false);
  const [deviceType, setDeviceType] = useState(null);

  useEffect(() => {
    // Check if banner was already dismissed
    const bannerDismissed = cookieStorage.getItem('appBannerDismissed');
    
    // Detect device type
    const userAgent = navigator.userAgent || navigator.vendor || window.opera;
    
    let device = null;
    if (/iPad|iPhone|iPod/.test(userAgent) && !window.MSStream) {
      device = 'ios';
    } else if (/android/i.test(userAgent)) {
      device = 'android';
    }
    
    // Check if it's a mobile device
    const isMobile = /Android|webOS|iPhone|iPad|iPod|BlackBerry|IEMobile|Opera Mini/i.test(userAgent);
    
    // Show banner only if:
    // 1. It's a mobile device
    // 2. Banner wasn't dismissed
    // 3. Not already in app (check if there's a specific user agent for your app)
    if (isMobile && !bannerDismissed && device) {
      setDeviceType(device);
      setShowBanner(true);
    }
  }, []);

  const handleClose = () => {
    setShowBanner(false);
    // Save dismissal state for 30 days
    cookieStorage.setItem('appBannerDismissed', 'true', { expires: 30 });
  };

  const handleInstallClick = () => {
    if (deviceType === 'ios') {
      // Replace with your actual App Store link
      window.open('https://apps.apple.com/app/alghazal/YOUR_APP_ID', '_blank');
    } else if (deviceType === 'android') {
      // Replace with your actual Google Play link
      window.open('https://play.google.com/store/apps/details?id=com.alghazal.app', '_blank');
    }
    handleClose();
  };

  if (!showBanner) return null;

  return (
    <>
      {/* Fixed Banner at Top */}
      <div className="fixed top-0 left-0 right-0 z-[9999] bg-gradient-to-r from-[#162A2B] to-[#1a3a3b] shadow-lg animate-slideDown">
        <div className="relative px-4 py-3 flex items-center justify-between gap-3">
          {/* App Icon */}
          <div className="flex-shrink-0">
            <Image 
              src={img.logo} 
              alt="Alghazal App" 
              width={50} 
              height={50}
              className="rounded-xl"
            />
          </div>

          {/* Content */}
          <div className="flex-1 min-w-0">
            <h3 className="text-white font-bold text-sm mb-1">
              {t('app_banner.title') || 'تطبيق الغزال'}
            </h3>
            <p className="text-gray-200 text-xs line-clamp-1">
              {t('app_banner.description') || 'احصل على تجربة أفضل مع التطبيق'}
            </p>
          </div>

          {/* Install Button */}
          <button
            onClick={handleInstallClick}
            className="flex-shrink-0 bg-white text-[#162A2B] px-4 py-2 rounded-lg font-bold text-sm hover:bg-gray-100 transition-colors"
          >
            {t('app_banner.install') || 'تحميل'}
          </button>

          {/* Close Button */}
          <button
            onClick={handleClose}
            className="flex-shrink-0 text-white hover:text-gray-300 transition-colors p-1"
            aria-label="Close"
          >
            <CloseOutlined className="text-lg" />
          </button>
        </div>
      </div>

      {/* Spacer to prevent content from being hidden under banner */}
      <div className="h-[70px]" />

      <style jsx>{`
        @keyframes slideDown {
          from {
            transform: translateY(-100%);
            opacity: 0;
          }
          to {
            transform: translateY(0);
            opacity: 1;
          }
        }

        .animate-slideDown {
          animation: slideDown 0.3s ease-out;
        }
      `}</style>
    </>
  );
};

export default AppInstallBanner;

