'use client';

import { CalendarOutlined, EnvironmentOutlined, SearchOutlined } from '@ant-design/icons';
import { Button, Col, Form, Row, message } from 'antd';
import dayjs from 'dayjs';
import { useRouter } from 'next/navigation';
import { useEffect, useRef, useState } from 'react';
import { useDispatch, useSelector } from 'react-redux';

import Icon from '@/constants/icon';
import useFormSessionStorage from '@/hook/useFormSessionStorage';
import cookieStorage from '@/utils/cookieStorage';
import { CustomDatePicker, CustomDropdown, CustomInput } from '@components/ui';
import { setReceptionData } from '@store/features/lim/limousineSlice';
import { useTranslations } from 'next-intl';

const LimReceptionPricing = ({ locale }) => {
    const [form] = Form.useForm();
    const [locationType, setLocationType] = useState('Pickup_Airport');
    const [isSearching, setIsSearching] = useState(false);
    const [selectedPickupLocation, setSelectedPickupLocation] = useState(null);
    const [selectedDropoffLocation, setSelectedDropoffLocation] = useState(null);
    const dispatch = useDispatch();
    const router = useRouter();
    const searchTimeoutRef = useRef(null);
    const t = useTranslations();

    // استخدام hook لحفظ واسترجاع البيانات من session
    const { saveToSession, getSavedData } = useFormSessionStorage(
        'lim_reception_form',
        form,
        { locationType, selectedPickupLocation, selectedDropoffLocation }
    );

    // استرجاع البيانات المحفوظة عند التحميل
    useEffect(() => {
        const savedData = getSavedData();
        if (savedData) {
            // استرجاع نوع الموقع
            if (savedData.dependencies?.locationType) {
                setLocationType(savedData.dependencies.locationType);
            }
            
            // استرجاع المواقع المختارة
            if (savedData.dependencies?.selectedPickupLocation) {
                setSelectedPickupLocation(savedData.dependencies.selectedPickupLocation);
            }
            if (savedData.dependencies?.selectedDropoffLocation) {
                setSelectedDropoffLocation(savedData.dependencies.selectedDropoffLocation);
            }

            // استرجاع قيم النموذج
            if (savedData.formValues) {
                const formValues = { ...savedData.formValues };
                // تحويل التاريخ من ISO string إلى dayjs object
                if (formValues.receptionDateTime) {
                    formValues.receptionDateTime = dayjs(formValues.receptionDateTime);
                }
                form.setFieldsValue(formValues);
            }
        }
    }, []);

    // حفظ البيانات عند تغيير أي قيمة في النموذج
    useEffect(() => {
        const handleFormChange = () => {
            saveToSession({
                locationType,
                selectedPickupLocation,
                selectedDropoffLocation
            });
        };

        // إضافة مستمع لتغيرات النموذج
        const interval = setInterval(() => {
            const values = form.getFieldsValue();
            if (values && Object.keys(values).length > 0) {
                handleFormChange();
            }
        }, 1000); // حفظ كل ثانية إذا كان هناك تغيير

        return () => clearInterval(interval);
    }, [locationType, selectedPickupLocation, selectedDropoffLocation, form, saveToSession]);

    // Cleanup timeout on unmount
    useEffect(() => {
        return () => {
            if (searchTimeoutRef.current) {
                clearTimeout(searchTimeoutRef.current);
            }
        };
    }, []);

    // حفظ البيانات عند تغيير نوع الموقع
    useEffect(() => {
        // لا نمسح المواقع المختارة عند التغيير، فقط نحفظ التغيير
        saveToSession({
            locationType,
            selectedPickupLocation,
            selectedDropoffLocation
        });
    }, [locationType]);

    const onFinish = (values) => {
        try {
            // منع البحث إذا كان قيد التنفيذ
            if (isSearching) return;

            // التحقق من أن أحد الموقعين يجب أن يكون مطار جدة - فقط في حالة اختيار المطار
            if (locationType === 'Pickup_Airport') {
                const hasAirportPickup = values.to === 'JEDDAH_AIRPORT';
                const hasAirportDropoff = values.from === 'JEDDAH_AIRPORT';
                
                if (!hasAirportPickup && !hasAirportDropoff) {
                    // إظهار رسالة خطأ
                    message.error(t('airport_required_error'));
                    return;
                }
            }

            // بدء حالة التحميل
            setIsSearching(true);

            const payload = {
                ...values,
                typeLim: locationType,
                receptionDateTime: values.receptionDateTime?.toISOString() || null,
                additionalHoursPerDay: values.additionalHoursPerDay || 0,
            };

            // حفظ البيانات
            cookieStorage.setItem('receptionData', JSON.stringify(payload));
            dispatch(setReceptionData(payload));

            const filtered = Object.fromEntries(
                Object.entries(payload).filter(([_, v]) => v != null)
            );
            const queryString = new URLSearchParams(filtered).toString();

            // إضافة تأخير للـ UX مع timeout للحماية
            searchTimeoutRef.current = setTimeout(() => {
                try {
                    // توجيه إلى نفس الصفحة مع البيانات
                    router.push(`${locale}/limo?${queryString}`);
                } catch (error) {
                    console.error('Error navigating to search results:', error);
                    setIsSearching(false);
                }
            }, 800); // تأخير 800ms لإظهار loading

            // إيقاف التحميل بعد 3 ثوان كحد أقصى
            setTimeout(() => {
                setIsSearching(false);
            }, 3000);

        } catch (error) {
            console.error('Error during search:', error);
            setIsSearching(false);
        }
    };

    // تكوين خيارات مكان الاستقبال
    const getPickupLocationOptions = () => {
        const baseOptions = [
            { value: 'JEDDAH', label: t('from_jeddah') },
            { value: 'MAKKAH', label: t('from_makkah') },
        ];

        if (locationType === 'Pickup_Airport') {
            // إضافة خيار المطار ولكن تعطيله إذا كان مكان النزول مطار
            baseOptions.unshift({
                value: 'JEDDAH_AIRPORT',
                label: t('from_airport'),
                disabled: selectedDropoffLocation === 'JEDDAH_AIRPORT'
            });
        }

        return baseOptions;
    };

    // تكوين خيارات مكان النزول
    const getDropoffLocationOptions = () => {
        const baseOptions = [
            { value: 'JEDDAH', label: t('to_jeddah') },
            { value: 'MAKKAH', label: t('to_makkah') },
        ];

        if (locationType === 'Pickup_Airport') {
            // إضافة خيار المطار ولكن تعطيله إذا كان مكان الاستقبال مطار
            baseOptions.unshift({
                value: 'JEDDAH_AIRPORT',
                label: t('to_airport'),
                disabled: selectedPickupLocation === 'JEDDAH_AIRPORT'
            });
        }

        return baseOptions;
    };
    const { settings } = useSelector((state) => state.shared);
 
    return (
        <Form 
            form={form} 
            onFinish={onFinish}
            initialValues={{
                to: undefined,
                from: undefined,
                receptionDateTime: undefined,
                flightNumber: undefined
            }}
        >
            <Row gutter={[16, 0]} className="mt-6 max-sm:mt-0">
                {/* أزرار اختيار نوع الموقع */}
                <Col xs={24} sm={24} md={4} lg={2} className='max-sm:mb-5'>
                    <div className="flex flex-col gap-2 max-sm:flex-row h-full">
                        <Button
                            size="small"
                            className={`w-full max-sm:w-auto  max-sm:px-4 m-0 p-0 text-md border rounded-none transition-all duration-300 [&_svg]:fill-current text-[10px]
              ${locationType === 'Pickup_Airport'
                                    ? '!bg-[#162A2B] !text-[#fff] !border-[#162A2B]'
                                    : '!bg-white !text-[#162A2B] !border-[#E0A939]'}`}
                            onClick={() => setLocationType('Pickup_Airport')}
                            icon={<Icon.AirIcons />}
                        >
                            {t('limousine.airport')}
                        </Button>

                        <Button
                            size="small"
                            className={`w-full max-sm:w-auto max-sm:px-4 text-md border rounded-none transition-all duration-300 [&_svg]:fill-current text-[10px]
              ${locationType === 'Pickup_OtherPlaces'
                                    ? '!bg-[#162A2B] !text-[#fff] !border-[#162A2B]'
                                    : '!bg-white !text-[#162A2B] !border-[#E0A939]'}`}
                            onClick={() => setLocationType('Pickup_OtherPlaces')}
                            icon={<EnvironmentOutlined />}
                        >
                            {t('limousine.other_places')}
                        </Button>
                    </div>
                </Col>

                {/* مكان الاستقبال */}
                <Col xs={12} sm={12} md={8} lg={4}>
                    <CustomDropdown
                        name="to"
                        options={getPickupLocationOptions()}
                        placeholder={t('limousine.pickup_location')}
                        className="w-full h-[50px] max-sm:h-[40px] [&_.ant-select-selector]:max-sm:!text-xs border border-[#162A2B] rounded text-sm"
                        allowClear
                        required={false}
                        onChange={(value) => {
                            setSelectedPickupLocation(value);
                            // إذا تم اختيار المطار كمكان استقبال، امسح مكان النزول إذا كان مطار
                            if (value === 'JEDDAH_AIRPORT' && selectedDropoffLocation === 'JEDDAH_AIRPORT') {
                                form.setFieldValue('from', undefined);
                                setSelectedDropoffLocation(null);
                            }
                        }}
                    />
                </Col>

                {/* مكان النزول */}
                <Col xs={12} sm={12} md={8} lg={4}>
                    <CustomDropdown
                        name="from"
                        options={getDropoffLocationOptions()}
                        placeholder={t('limousine.dropoff_location')}
                        className="w-full h-[50px] max-sm:h-[40px] [&_.ant-select-selector]:max-sm:!text-xs   border border-[#162A2B] rounded text-sm"
                        allowClear
                        required={false}
                        onChange={(value) => {
                            setSelectedDropoffLocation(value);
                            // إذا تم اختيار المطار كمكان نزول، امسح مكان الاستقبال إذا كان مطار
                            if (value === 'JEDDAH_AIRPORT' && selectedPickupLocation === 'JEDDAH_AIRPORT') {
                                form.setFieldValue('to', undefined);
                                setSelectedPickupLocation(null);
                            }
                        }}
                    />
                </Col>

                {/* التاريخ والوقت */}
                <Col xs={12} sm={12} md={8} lg={6}>
                    <CustomDatePicker
                        name="receptionDateTime"
                        placeholder={t('limousine.date_time')}
                        size="large"
                        showTime
                        format="YYYY-MM-DD HH:mm"
                        className="w-full !h-[50px] border max-sm:!h-[40px] [&_.ant-picker-input>input]:max-sm:!text-xs border-[#162A2B] rounded text-sm"
                        suffixIcon={<CalendarOutlined />}
                        maxDaysFromNow={settings?.["NUM_HOURS_FOR_START_RESERVATION_LIMOUSINE"]  }

                    />
                </Col>

                {/* رقم الرحلة - يظهر فقط للمطار */}
                {locationType === 'Pickup_Airport' && (
                    <Col xs={12} sm={12} md={8} lg={4}>

                        <CustomInput
                            placeholder={t('limousine.flight_number')}
                            className="w-full !h-[50px] border max-sm:!h-[40px]  max-sm:!text-xs !border-[#162A2B] !shadow-none rounded text-sm"
                            size="large" name="flightNumber"
                        />
                    </Col>
                )}




                {/* زر البحث */}
                <Col xs={12} sm={24} md={8} lg={4}>
                    <Button
                        size="large"
                        type="primary"
                        loading={isSearching}
                        disabled={isSearching}
                        className="w-full h-[50px] max-sm:!h-[40px] [&_>span]:max-sm:!text-sm   !bg-[#162A2B] !text-white flex items-center justify-center gap-1 min-w-[120px]"
                        onClick={() => form.submit()}
                    >
                        {!isSearching && <SearchOutlined />}
                        {isSearching ? t('search.searching') : t('search.search_results')}
                    </Button>
                </Col>
            </Row>
        </Form >
    );
};

export default LimReceptionPricing;