'use client';

import { useSessionCleanup } from '@/hook/useSessionCleanup';
import { setDuration } from '@/store/features/rental/rentalSlice';
import dayjs from 'dayjs';
import 'dayjs/locale/ar';
import { useLocale } from 'next-intl';
import { usePathname, useRouter, useSearchParams } from 'next/navigation';
import { useEffect, useState } from 'react';
import { useDispatch } from 'react-redux';
import './filter.css';
import FiltersSkeleton from './FiltersSkeleton';
import CompanyForm from './forms/CompanyForm';
import LimousineForm from './forms/lim/LimousineForm';
import RentalForm from './forms/rental/RentalForm';
import DurationSelector from './Tabs/DurationSelector';
import PickupDeliverySelector from './Tabs/PickupDeliverySelector';
import TabsHeader from './Tabs/TabsHeader';

dayjs.locale('ar');

const FiltersCars = () => {
  const locale = useLocale();
  const searchParams = useSearchParams();
  const router = useRouter();
  const pathname = usePathname();
  const dispatch = useDispatch();
  const [activeTab, setActiveTab] = useState('rental'); 
  const [isLoading, setIsLoading] = useState(true);
  const { clearRentalBookingDataWithStore } = useSessionCleanup()

  // Function to handle tab change and update URL
  const handleTabChange = (tab) => {
    setActiveTab(tab);
    
    // Update URL with new tab parameter
    const params = new URLSearchParams(searchParams.toString());
    params.set('tab', tab);
    router.replace(`${pathname}?${params.toString()}`, { scroll: false });
  };

  // Clear data only on initial mount
  useEffect(() => {
    // Get duration from URL before clearing
    const urlDuration = searchParams.get('duration');
    
    clearRentalBookingDataWithStore()
    
    // Restore duration from URL if it exists and is valid
    if (urlDuration && ['daily', 'weekly', 'monthly'].includes(urlDuration)) {
      dispatch(setDuration(urlDuration));
    }
    
    // Remove artificial loading delay to improve CLS
    setIsLoading(false);
  }, [])

  // Check for tab parameter in URL whenever it changes
  useEffect(() => {
    const tabParam = searchParams.get('tab');
    if (tabParam && ['rental', 'companies', 'limousine'].includes(tabParam)) {
      setActiveTab(tabParam);
    }
  }, [searchParams])

  // Show skeleton while loading
  if (isLoading) {
    return <FiltersSkeleton />;
  }

  return (
    <div className="app_filter_cars !mt-[-150px] max-md:!mt-[-100px] mx-auto lg:p-4 md:p-0 max-md:px-4">
      {/* Tabs */}
      <TabsHeader activeTab={activeTab} setActiveTab={handleTabChange} />

      {/* Main Container */}
      <div className="main_content bg-white sticky shadow-md rounded-b-xl px-4 md:px-6 pt-6 md:pt-8 pb-6 md:pb-[28px] space-y-4">
        {/* Row 1: Pickup/Delivery + Duration */}
        <div className="flex flex-wrap flex-col gap-4">
          {activeTab === 'rental' && <>
            <PickupDeliverySelector />
            <DurationSelector />
          </>
          }
        </div>

        {/* Conditional Forms */}
        {activeTab === 'rental' && <RentalForm />}
        {activeTab === 'companies' && <CompanyForm />}
        {activeTab === 'limousine' && <LimousineForm locale={locale} />}
      </div>
    </div>
  );
};

export default FiltersCars;
